﻿Public Class cl_Irinos
    Inherits cl_MscDll

    Public Const MSC_BUF_SIZE_STATIC As UInt32 = 1500
    Public Const MSC_BUF_SIZE_BITIO As UInt32 = 64
    Public Const MSC_BUF_SIZE_HARDSTAT As UInt32 = 256
    Public Const MSC_BUF_SIZE_WRITECMD As UInt32 = 1000
    Public Const MAX_DYNAMIC_MEASUREMENT As UInt32 = 2
    Public Const MAX_DYN_CHANNELS As UInt32 = 32

    Public Const WM_USER As UInt32 = &H400


    Enum E_CONNECT_STATUS
        cscUndefined        'Initialization
        cscTryConnect       'Trying to connect
        cscNoDevices        'No devices found
        cscNoDeviceInfo     'No device info available
        cscConnected        'Connection established successfully
        cscConnectionClosed 'Connection closed
        cscNoDeviceOpen     'Failed to open connection to device
        cscInitDone         'Device initialization finished
        cscInitFailed       'Device initialization failed
        cscCommStarted      'Communication started
        cscCommFailed       'Failed to start communication
    End Enum

    'DLL communication opcodes
    Enum E_MSC_OPCODES As Byte
        'Initialization
        opcRIV = &H1        'Read inventory
        opcRMI = &H3        'Read module information
        opcRSS = &H5        'Read system string
        opcWCC = &H9        'Write channel characteristics
        opcRCA = &H10       'Read channel assignment
        opcWCA = &H11       'Write channel assignment
        opcWCP = &H1E       'Set communication parameter

        'Configuration, various
        opcWCL = &H22       'Write channel list
        opcRCL = &H23       'Read channel list
        opcAL = &H24        'Activate channel list for static measurement
        opcDT = &H30        'Define trigger for dynamic measurement
        opcAT = &H31        'Activate trigger for dynamic measurement
        opcIT = &H32        'Inactivate trigger for dynamic measurement
        opcSP = &H35        'Set (channel) parameter
        opcRHS = &H38       'Read hardware status
        opcREv = &H39       'Read event status
        opcSAbsT = &H3A     'Set absolute time
        opcREvCfg = &H3C    'Read event configuration and status
        opcWEvCfg = &H3D    'Write event configuration

        'Measurement
        opcRS = &H40        'Read static measurement values
        opcBIO = &H42       'Read digital inputs / write digital outputs
        opcRSW = &H44       'Read status word for dynamic measurements
        opcDDM1 = &H50      'Define dynamic measurement 1
        opcDDM2 = &H51      'Define dynamic measurement 2
        opcRDM1 = &H60      'Read data of dynamic measurement 1
        opcRDM2 = &H61      'Read data of dynamic measurement 2

        'Service
        opcRst = &H7E       'System-Reset
    End Enum


    '###########################################################################
    'USAGE OF BUFFERS
    '----------------
    'This example was written for a .NET-based visual basic environment.
    'In a .NET based environment, a data buffer is an object, which can be
    'moved by the .NET framework in memory automatically. Hence also the memory
    'addresses change dynamically.
    'The buffers are assigned to the DLL via pointers, which are similar to
    'ByRef in VisualBasic. However, working with pointers only works, if
    'memory addresses do not change.
    'In order to get buffers with fixed addresses, the .NET-class
    'System.Runtime.InteropServices.Marshal is used.
    'If an older version of visual basic is used, simple byte arrays can be used
    'instead.
    '###########################################################################


    Public eConnStat As E_CONNECT_STATUS = E_CONNECT_STATUS.cscUndefined
    Dim pDevice As UInt32 = 0
    Dim pucSndBufferStatic As IntPtr = System.Runtime.InteropServices.Marshal.AllocHGlobal(1)
    Dim pucSndBufferBitIO As IntPtr = System.Runtime.InteropServices.Marshal.AllocHGlobal(CInt(MSC_BUF_SIZE_BITIO))
    Dim pucSndBufferHardStat As IntPtr = System.Runtime.InteropServices.Marshal.AllocHGlobal(1)
    Dim pucRcvBufferCmd As IntPtr = System.Runtime.InteropServices.Marshal.AllocHGlobal(CInt(MSC_BUF_SIZE_WRITECMD))
    Dim pucSndBufferCmd As IntPtr = System.Runtime.InteropServices.Marshal.AllocHGlobal(CInt(MSC_BUF_SIZE_WRITECMD))
    Dim pucDummy As IntPtr = System.Runtime.InteropServices.Marshal.AllocHGlobal(1)
    Dim aucCmd0(MSC_BUF_SIZE_WRITECMD) As Byte
    Dim aulDynBytesPerChannel(MAX_DYNAMIC_MEASUREMENT) As UInt32
    Dim apucValBufDyn(MAX_DYNAMIC_MEASUREMENT, MAX_DYN_CHANNELS) As IntPtr


    '******************************************************************************
    ' FUNCTION: cl_Irinos::ConnectIrinos
    '-----------------------------------------------------------------------------
    ' Trying to establish a connection to the first Irinos-System, which can be
    ' found. Communication parameters are taken from Msc.cfg by the DLL.
    '
    ' Note:
    ' It is possible to access more than one Irinos-System from the same PC.
    ' In this case more than one Device-Handle must be created.
    ' However: In more than 10 years, there was not any application with this
    ' requirement.
    '-----------------------------------------------------------------------------
    ' PARAMETERS: none
    '-----------------------------------------------------------------------------
    ' Return: see type definition
    '*****************************************************************************
    Function ConnectIrinos() As E_CONNECT_STATUS
        Dim eResult As E_CONNECT_STATUS = E_CONNECT_STATUS.cscTryConnect
        Dim ulNDevices As UInt32 = 0
        Dim ulBusType As UInt32 = 0
        Dim acStr(MSC_MAX_UNIQUEID_SIZE) As Byte
        Dim ulResult As UInt32 = 0

        For I = 0 To (MSC_BUF_SIZE_WRITECMD - 1)
            aucCmd0(I) = 0
        Next

        For I = 0 To (MAX_DYNAMIC_MEASUREMENT - 1)
            aulDynBytesPerChannel(I) = 0
            For J = 0 To (MAX_DYN_CHANNELS - 1)
                apucValBufDyn(I,J) = 0
            Next
        Next

        'Get count of available devices.
        ' If 1 Irinos-System is connected, 1 should be returned.
        ulResult = F_MSC_EnumerateDevices(ulNDevices)
        If ulResult <> MSC_STATUS_SUCCESS Then
            eResult = E_CONNECT_STATUS.cscNoDevices
        End If

        'Read device information
        If eResult = E_CONNECT_STATUS.cscTryConnect Then
            ulBusType = 0
            If F_MSC_GetDeviceInfo(ulNDevices - 1, ulBusType, acStr(0)) <> MSC_STATUS_SUCCESS Then
                eResult = E_CONNECT_STATUS.cscNoDeviceInfo
            End If
        End If

        'Open device / create handle for device
        If eResult = E_CONNECT_STATUS.cscTryConnect Then
            pDevice = 0
            If F_MSC_OpenDevice(ulNDevices - 1, pDevice) = MSC_STATUS_SUCCESS Then
                eResult = E_CONNECT_STATUS.cscConnected
            Else
                eResult = E_CONNECT_STATUS.cscNoDeviceOpen
            End If
        End If

        'Initialize device
        If eResult = E_CONNECT_STATUS.cscConnected Then
            If F_MSC_InitDevice(pDevice) = MSC_STATUS_SUCCESS Then
                eResult = E_CONNECT_STATUS.cscInitDone
            Else
                eResult = E_CONNECT_STATUS.cscInitFailed
            End If
        End If

        'Start communication
        If eResult = E_CONNECT_STATUS.cscInitDone Then
            If F_MSC_Start(pDevice, 1, 500, 10, 75) = MSC_STATUS_SUCCESS Then
                eResult = E_CONNECT_STATUS.cscCommStarted
            Else
                eResult = E_CONNECT_STATUS.cscCommFailed
            End If
        End If

        eConnStat = eResult

        Return eResult
    End Function


    '******************************************************************************
    ' FUNCTION: cl_Irinos::StopAndClose
    '-----------------------------------------------------------------------------
    ' Close the connection (if opened).
    '-----------------------------------------------------------------------------
    ' PARAMETERS: none
    '-----------------------------------------------------------------------------
    ' Return: true if a connection was closed.
    '*****************************************************************************
    Function StopAndClose() As Boolean
        If pDevice <> 0 Then
            F_MSC_Stop(pDevice)
            F_MSC_CloseDevice(pDevice)
            eConnStat = E_CONNECT_STATUS.cscConnectionClosed
            Return True
        Else
            Return False
        End If
    End Function


    '******************************************************************************
    ' FUNCTION: cl_Irinos::SetupStaticChannel
    '-----------------------------------------------------------------------------
    ' Start reading static measurement values / Bit I/O / hardware status
    '-----------------------------------------------------------------------------
    ' PARAMETERS:
    ' ucOpode: DLL-Opcode opcRS or opcBIO or opcRHS.
    ' ulSendSize: Size of data to be sent from corresponding send buffer.
    '             --> currently only required for opcBIO!
    ' pWindowHandle: Window handle which shall receive the notification message,
    '                when the static channel was updated.
    '-----------------------------------------------------------------------------
    ' Return: true if a static channel was setup successfully.
    '*****************************************************************************
    Function SetupStaticChannel(ByVal eOpcode As E_MSC_OPCODES, ByVal ulSendSize As UInt32, ByVal pWindowsHandle As IntPtr) As Boolean
        Dim bReturnValue As Boolean = False

        If eConnStat = E_CONNECT_STATUS.cscCommStarted Then
            bReturnValue = True

            Select Case eOpcode
                Case E_MSC_OPCODES.opcRS    '## Read static measurement values
                    System.Runtime.InteropServices.Marshal.WriteByte(pucSndBufferStatic, 0)
                    'Register static channel
                    If F_MSC_SetupStaticChannel(pDevice, E_MSC_OPCODES.opcRS, 1, pucSndBufferStatic, MSC_BUF_SIZE_STATIC) <> MSC_STATUS_SUCCESS Then
                        bReturnValue = False
                        'Register  Message "on response"
                    ElseIf F_MSC_SetNotificationMessage(pDevice, E_MSC_OPCODES.opcRS, pWindowsHandle, WM_USER, 10, 10) <> MSC_STATUS_SUCCESS Then
                        bReturnValue = False
                        'Register Message "on error"
                    ElseIf F_MSC_SetNotificationMessage(pDevice, -1, pWindowsHandle, WM_USER, 20, 20) <> MSC_STATUS_SUCCESS Then
                        bReturnValue = False
                    End If

                Case E_MSC_OPCODES.opcBIO   '## Read digital inputs / write digital outputs
                    If ulSendSize > MSC_BUF_SIZE_BITIO Then
                        ulSendSize = MSC_BUF_SIZE_BITIO
                    End If

                    Dim aucBitIOSndBufInit(MSC_BUF_SIZE_BITIO) As Byte
                    For I = 0 To (MSC_BUF_SIZE_BITIO - 1)
                        aucBitIOSndBufInit(I) = 0
                    Next
                    System.Runtime.InteropServices.Marshal.Copy(aucBitIOSndBufInit, 0, pucSndBufferBitIO, MSC_BUF_SIZE_BITIO)

                    'Register static channel
                    If F_MSC_SetupStaticChannel(pDevice, E_MSC_OPCODES.opcBIO, ulSendSize, pucSndBufferBitIO, MSC_BUF_SIZE_BITIO) <> MSC_STATUS_SUCCESS Then
                        bReturnValue = False
                        'Register Message "on response"
                    ElseIf F_MSC_SetNotificationMessage(pDevice, E_MSC_OPCODES.opcBIO, pWindowsHandle, WM_USER, 30, 30) <> MSC_STATUS_SUCCESS Then
                        bReturnValue = False
                    End If

                Case E_MSC_OPCODES.opcRHS   '## Read hardware status
                    System.Runtime.InteropServices.Marshal.WriteByte(pucSndBufferHardStat, 2)
                    'Register static channel
                    If F_MSC_SetupStaticChannel(pDevice, E_MSC_OPCODES.opcRHS, 1, pucSndBufferHardStat, MSC_BUF_SIZE_HARDSTAT) <> MSC_STATUS_SUCCESS Then
                        bReturnValue = False
                        'Register Message "on response"
                    ElseIf F_MSC_SetNotificationMessage(pDevice, E_MSC_OPCODES.opcRHS, pWindowsHandle, WM_USER, 40, 40) <> MSC_STATUS_SUCCESS Then
                        bReturnValue = False
                    End If

                Case Else   '## Opcode not supported
                    bReturnValue = False
            End Select

        End If

        Return bReturnValue
    End Function



    '******************************************************************************
    ' FUNCTION: cl_Irinos::GetStaticMeasurementValues
    '-----------------------------------------------------------------------------
    ' Copy the static measurement values into the buffer given in the parameters.
    ' This function is typically called after new measurement values have been
    ' received, e.g. to update the GUI.
    '-----------------------------------------------------------------------------
    ' PARAMETERS:
    ' aslValueBuffer: Static measurement values will be copied into this array.
    '-----------------------------------------------------------------------------
    ' Return: nA
    '*****************************************************************************/
    Sub GetStaticMeasurementValues(ByRef aslValueBuffer() As Int32)
        Dim ulDataCount As UInt32 = 0

        If eConnStat = E_CONNECT_STATUS.cscCommStarted Then
            'Request values from the DLL
            F_MSC_ReadStatic32(pDevice, E_MSC_OPCODES.opcRS, aslValueBuffer.Length * 4, aslValueBuffer(0), ulDataCount)
        End If
    End Sub


    '******************************************************************************
    ' FUNCTION: cl_Irinos::GetHardStat
    '-----------------------------------------------------------------------------
    ' Copy the hardware status data into the buffer given in the parameters.
    ' This function is typically called after new hardware status data has been
    ' received, e.g. to update the GUI.
    '-----------------------------------------------------------------------------
    ' PARAMETERS:
    ' aucHardStat: Hardware status data will be copied into this array.
    '-----------------------------------------------------------------------------
    ' Return: nA
    '*****************************************************************************/
    Sub GetHardStat(ByRef aucHardStat() As Byte)
        Dim ulDataCount As UInt32 = 0

        If eConnStat = E_CONNECT_STATUS.cscCommStarted Then
            'Request hardware status from the DLL
            F_MSC_ReadStatic(pDevice, E_MSC_OPCODES.opcRHS, aucHardStat.Length, aucHardStat(0), ulDataCount)
        End If
    End Sub


    '******************************************************************************
    ' FUNCTION: cl_Irinos::GetBitIO
    '-----------------------------------------------------------------------------
    ' Copy the bit data into the buffer given in the parameters.
    ' This function is typically called after new Bit I/O data has been
    ' received, e.g. to update the GUI.
    '-----------------------------------------------------------------------------
    ' PARAMETERS:
    ' aucBitData: Bit I/O data will be copied into this array.
    '-----------------------------------------------------------------------------
    ' Return: nA
    '*****************************************************************************/
    Sub GetBitIO(ByRef aucBitData() As Byte)
        Dim ulDataCount As UInt32 = 0

        If eConnStat = E_CONNECT_STATUS.cscCommStarted Then
            'Request input data from the DLL
            F_MSC_ReadStatic(pDevice, E_MSC_OPCODES.opcBIO, aucBitData.Length, aucBitData(0), ulDataCount)
        End If
    End Sub


    '******************************************************************************
    ' FUNCTION: cl_Irinos::RefreshOutputs
    '-----------------------------------------------------------------------------
    ' Copy the output data into the Bit I/O send buffer and tell the MscDll that
    'the data has been updated.
    '-----------------------------------------------------------------------------
    ' PARAMETERS:
    ' aucOutputData: Output Bits (1 Byte per 8 Bits)
    '-----------------------------------------------------------------------------
    ' Return: nA
    '*****************************************************************************
    Sub RefreshOutputs(ByVal aucOutputData() As Byte)
        Dim ulBytes As UInt32 = 0

        If eConnStat = E_CONNECT_STATUS.cscCommStarted Then
            ulBytes = aucOutputData.Length
            If (ulBytes > MSC_BUF_SIZE_BITIO) Then
                ulBytes = MSC_BUF_SIZE_BITIO
            End If

            System.Runtime.InteropServices.Marshal.Copy(aucOutputData, 0, pucSndBufferBitIO, ulBytes)

            F_MSC_RefreshChannel(pDevice, E_MSC_OPCODES.opcBIO)
        End If
    End Sub


    '******************************************************************************
    ' FUNCTION: cl_Irinos::WriteCommand
    '-----------------------------------------------------------------------------
    ' Write a command to the Irinos-system (and receive the corresponding
    ' response).
    ' Data written to aucSndBufferCmd will be sent.
    ' Received data will be written to aucRcvBufferCmd.
    '-----------------------------------------------------------------------------
    ' PARAMETERS:
    ' eOpcode: Opcode of command, which shall be sent. E.g. opcRMI to read
    '          module information.
    ' aucSendData: Buffer with data to send
    ' aucReceiveData: Buffer with data to receive
    ' pulReceiveSize: Number of received bytes written into aucReceiveData.
    '-----------------------------------------------------------------------------
    ' Return: true if Command was executed successfully.
    '*****************************************************************************
    Function WriteCommand(ByVal eOpcode As E_MSC_OPCODES, ByRef aucSendData() As Byte, ByRef aucReceiveData() As Byte, ByRef pulReceiveSize As UInt32)
        Dim bReturnValue As Boolean = False
        Dim ulTxLength As UInt32 = 0

        If eConnStat = E_CONNECT_STATUS.cscCommStarted Then
            System.Runtime.InteropServices.Marshal.Copy(aucCmd0, 0, pucRcvBufferCmd, MSC_BUF_SIZE_WRITECMD)
            System.Runtime.InteropServices.Marshal.Copy(aucCmd0, 0, pucSndBufferCmd, MSC_BUF_SIZE_WRITECMD)

            ulTxLength = aucSendData.Length
            If ulTxLength > MSC_BUF_SIZE_WRITECMD Then
                ulTxLength = MSC_BUF_SIZE_WRITECMD
            End If
            System.Runtime.InteropServices.Marshal.Copy(aucSendData, 0, pucSndBufferCmd, ulTxLength)

            If F_MSC_WriteCommand(pDevice, eOpcode, ulTxLength, pucSndBufferCmd, MSC_BUF_SIZE_WRITECMD, pucRcvBufferCmd, pulReceiveSize, 500) = MSC_STATUS_SUCCESS Then
                bReturnValue = True
                If pulReceiveSize > aucReceiveData.Length Then
                    pulReceiveSize = aucReceiveData.Length
                End If
                System.Runtime.InteropServices.Marshal.Copy(pucRcvBufferCmd, aucReceiveData, 0, pulReceiveSize)
            End If
        End If
        Return bReturnValue
    End Function


    '******************************************************************************
    ' FUNCTION: WriteCommandStr
    '-----------------------------------------------------------------------------
    ' Write a string command to the Irinos system via the MscDll and receive the
    ' corresponding response string.
    ' The Irinos system works with 8-Bit ANSII-strings, while .net works with
    ' 16-Bit unicode-strings. Hence the encoding has to be changed.
    ' If transferring the string fails, the original string refString remains
    ' unchanged.
    ' NOTE: For almost all string based commands, the result string contains
    '       a result value. If a problem occurs, the value is typically negative
    '       (e.g. #-99#). Consult the documentation about the opcode for
    '       further information.
    '       This result string is not evaluated here!
    '-----------------------------------------------------------------------------
    ' PARAMETERS: 
    ' eOpcode: Irinos opcode. See type definition.
    ' refString: -> input: string to send
    '            <- output: received string (if transfer was successful)
    '-----------------------------------------------------------------------------
    ' Return: true if the data transfer was successful.
    '*****************************************************************************
    Function WriteCommandStr(ByVal eOpcode As E_MSC_OPCODES, ByRef str As String)
        Dim bReturnValue As Boolean = False
        Dim acByteData() As Byte
        Dim ulReceiveSize As UInt32 = 0
        Dim ulTxLength As UInt32 = 0
        Dim aucRcvBufferCmd(MSC_BUF_SIZE_WRITECMD) As Byte

        System.Runtime.InteropServices.Marshal.Copy(aucCmd0, 0, pucRcvBufferCmd, MSC_BUF_SIZE_WRITECMD)
        System.Runtime.InteropServices.Marshal.Copy(aucCmd0, 0, pucSndBufferCmd, MSC_BUF_SIZE_WRITECMD)

        acByteData = System.Text.Encoding.ASCII.GetBytes(str)
        ulTxLength = acByteData.Length
        If ulTxLength > MSC_BUF_SIZE_WRITECMD Then
            ulTxLength = MSC_BUF_SIZE_WRITECMD
        End If
        System.Runtime.InteropServices.Marshal.Copy(acByteData, 0, pucSndBufferCmd, ulTxLength)

        'Send command
        If F_MSC_WriteCommand(pDevice, eOpcode, ulTxLength, pucSndBufferCmd, MSC_BUF_SIZE_WRITECMD, pucRcvBufferCmd, ulReceiveSize, 500) = MSC_STATUS_SUCCESS Then
            bReturnValue = True
            System.Runtime.InteropServices.Marshal.Copy(pucRcvBufferCmd, aucRcvBufferCmd, 0, ulReceiveSize)
            str = System.Text.Encoding.ASCII.GetString(aucRcvBufferCmd)
        End If

        Return bReturnValue
    End Function


    '******************************************************************************
    ' FUNCTION: cl_Irinos::SetupDynamicChannel
    '-----------------------------------------------------------------------------
    ' Allocates memory for dynamic measurement values and assigns this buffer
    ' to the MscDll. The MscDll will write received dynamic measurement values
    ' automatically into this buffer.
    '-----------------------------------------------------------------------------
    ' PARAMETERS:
    ' eOpcode: opcRDM1 or opcRDM2.
    ' ulMaxMeasValues: Maximum number of measurement values (per channel) to be
    '				   received.
    ' ulNChannels: Number of measurement channels.
    '-----------------------------------------------------------------------------
    ' Return: true if dynamic channel was setup successfully.
    '*****************************************************************************
    Function SetupDynamicChannel(ByVal eOpcode As E_MSC_OPCODES, ByVal ulMaxMeasValues As UInt32, ByVal ulNChannels As UInt32) As Boolean
        Dim bReturnValue As Boolean = False
        Dim ulDyn As UInt32 = 0


        If eConnStat = E_CONNECT_STATUS.cscCommStarted Then
            If eOpcode = E_MSC_OPCODES.opcRDM2 Then
                ulDyn = 1
            End If

            If ulNChannels > MAX_DYN_CHANNELS Then
                ulNChannels = MAX_DYN_CHANNELS
            End If

            aulDynBytesPerChannel(ulDyn) = ulMaxMeasValues * 4

            'Setup extended dynamic channel
            If F_MSC_SetupExtendedDynamicChannel(pDevice, eOpcode, CByte(ulNChannels), 1, pucDummy) = MSC_STATUS_SUCCESS Then
                bReturnValue = True

                'Delete old buffer, if it exists
                For J = 0 To (MAX_DYN_CHANNELS - 1)
                    If apucValBufDyn(ulDyn, J) <> 0 Then
                        System.Runtime.InteropServices.Marshal.FreeHGlobal(apucValBufDyn(ulDyn, J))
                    End If
                Next

                'Assign new buffer with required size
                For J = 0 To (MAX_DYN_CHANNELS - 1)
                    apucValBufDyn(ulDyn, J) = System.Runtime.InteropServices.Marshal.AllocHGlobal(CInt(ulMaxMeasValues * 4))
                Next
                aulDynBytesPerChannel(ulDyn) = ulMaxMeasValues * 4

                For I As Byte = 0 To (ulNChannels - 1)
                    If bReturnValue = True Then
                        If F_MSC_AttachSubChannelBuffer(pDevice, eOpcode, I, aulDynBytesPerChannel(ulDyn), apucValBufDyn(ulDyn, I)) <> MSC_STATUS_SUCCESS Then
                            bReturnValue = False
                        End If
                    End If
                Next
            End If
        End If


        Return bReturnValue
    End Function


    '******************************************************************************
    ' FUNCTION: cl_Irinos::GetDynBytesReceived
    '-----------------------------------------------------------------------------
    ' Returns the number of bytes that have been received so far for the dynamic
    ' measurement given in the opcode.
    ' Note:
    ' Each measurement value is a 32 Bit signed value (even if it is 16 or 8 bit
    ' value inside the Irinos system). To get the number of values received per
    ' channel, use the following formula:
    ' nValsPerCchannelReceived = nBytesReceived / 4.
    '-----------------------------------------------------------------------------
    ' PARAMETERS:
    ' eOpcode: opcRDM1 or opcRDM2.
    ' ulPosition: Number of bytes received per channel. This value is only
    '             changed if the return value is true.
    '-----------------------------------------------------------------------------
    ' Return: true if position was read successfully.
    '*****************************************************************************
    Function GetDynBytesReceived(ByVal eOpcode As E_MSC_OPCODES, ByRef pulPosition As UInt32) As Boolean
        Dim bReturnValue As Boolean = False

        If eConnStat = E_CONNECT_STATUS.cscCommStarted Then
            If F_MSC_GetPosition(pDevice, eOpcode, pulPosition) = MSC_STATUS_SUCCESS Then
                bReturnValue = True
            Else
                pulPosition = 0
            End If
        End If

        Return bReturnValue
    End Function


    '******************************************************************************
    ' FUNCTION: cl_Irinos::GetDynValue
    '-----------------------------------------------------------------------------
    ' Returns a dynamic measurement value, which has already been received.
    ' (Measurement values are stored by the MscDll in apslValBufDyn. This
    ' buffer is created dynamically in cl_Irinos::SetupDynamicChannel.)
    ' It is NOT checked, if the value has already been received. This should be
    ' done using cl_Irinos::GetDynBytesReceived before.
    '-----------------------------------------------------------------------------
    ' PARAMETERS:
    ' eOpcode: opcRDM1 or opcRDM2.
    ' ulPosition: Number of the measurement value, beginning at 0.
    '             (0 -> First value, 1 -> Second value, ...).
    ' ulChannel: Number of measurement channel.
    ' pslValue: Value, which is returned. If no value is available, pslValue
    '           will not be changed.
    '-----------------------------------------------------------------------------
    ' Return: false if the measurement value is outside the buffer range.
    '*****************************************************************************
    Function GetDynValue(ByVal eOpcode As E_MSC_OPCODES, ByVal ulPosition As UInt32, ByVal ulChannel As UInt32, ByRef pslValue As Int32) As Boolean
        Dim bReturnValue As Boolean = False
        Dim ulDyn As UInt32 = 0
        Dim ulPos As UInt32 = ulPosition * 4

        If eOpcode = E_MSC_OPCODES.opcRDM2 Then
            ulDyn = 1
        End If

        Try
            pslValue = System.Runtime.InteropServices.Marshal.ReadInt32(apucValBufDyn(ulDyn, ulChannel), CInt(ulPos))
            bReturnValue = True
        Catch ex As Exception
            bReturnValue = False
        End Try

        Return bReturnValue
    End Function

End Class
