﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using Multimedia;   // See MMTimer.cs


/*###########################################################################
 * USAGE OF BUFFERS / MARSHALLING
 * ----------------
 * This example was written for a .NET-based environment.
 * In a .NET based environment, a data buffer is an object, which can be
 * moved by the .NET framework in memory automatically. Hence also the memory
 * addresses change dynamically.
 * Using the MscDll, buffers are assigned to the DLL via pointers. However,
 * working with pointers only works, if memory addresses do not change.
 * 
 * In order to get buffers with fixed addresses, the .NET-class
 * System.Runtime.InteropServices.Marshal is used.
 * This allows using the "non .NET" MscDll in the .NET based C# environment.
*########################################################################*/



namespace MscDll_Demo_VCS_2010
{
    // Connection stati
    public enum E_CONNECT_STATUS
    {
        cscUndefined,        // Initialization
        cscTryConnect,       // Trying to connect
        cscNoDevices,        // No devices found
        cscNoDeviceInfo,     // No device info available
        cscConnected,        // Connection established successfully
        cscConnectionClosed, // Connection closed
        cscNoDeviceOpen,     // Failed to open connection to device
        cscInitDone,         // Device initialization finished
        cscInitFailed,       // Device initialization failed
        cscCommStarted,      // Communication started
        cscCommFailed        // Failed to start communication
    };

    // Opcodes used for the MscDll
    public enum E_MSC_OPCODES
    {
        /* Initialization */
        opcRIV = 0x01,		// Read inventory
        opcRMI = 0x03,		// Read module information
        opcRSS = 0x05,	    // Read system string
        opcWCC = 0x09,		// Write channel characteristics
        opcRCA = 0x10,		// Read channel assignment
        opcWCA = 0x11,		// Write channel assignment
        opcWCP = 0x1E,		// Set communication parameter

        /* Configuration, various */
        opcWCL = 0x22,		// Write channel list
        opcRCL = 0x23,		// Read channel list
        opcAL = 0x24,		// Activate channel list for static measurement
        opcDT = 0x30,		// Define trigger for dynamic measurement
        opcAT = 0x31,		// Activate trigger for dynamic measurement
        opcIT = 0x32,		// Inactivate trigger for dynamic measurement
        opcSP = 0x35,		// Set (channel) parameter
        opcRCI = 0x36,      // Read channel information
        opcRHS = 0x38,		// Read hardware status
        opcREv = 0x39,		// Read event status
        opcSAbsT = 0x3A,	// Set absolute time
        opcREvCfg = 0x3C,	// Read event configuration and status
        opcWEvCfg = 0x3D,	// Write event configuration

        /* Measurement */
        opcRS = 0x40,		// Read static measurement values
        opcBIO = 0x42,		// Read digital inputs / write digital outputs
        opcRSW = 0x44,		// Read status word for dynamic measurements
        opcDDM1 = 0x50,		// Define dynamic measurement 1
        opcDDM2 = 0x51,		// Define dynamic measurement 2
        opcRDM1 = 0x60,		// Read data of dynamic measurement 1
        opcRDM2 = 0x61,		// Read data of dynamic measurement 2

        /* Service */
        opcRst = 0x7E,		// System-Reset
    };


    public static class cl_IriConst
    {
        // Windows Messages for MSC_SetupNotificationMessage.
        // User-defined messages start at 0x0400.
        public const int WM_USER = 0x400;
        public const int WM_MESSAGE_MSC_READSTATIC = WM_USER + 0;    // Message for static measurement values
        public const int WM_MESSAGE_MSC_BITIO = WM_USER + 1;         // Message for BitIO
        public const int WM_MESSAGE_MSC_HW_STATUS = WM_USER + 2;     // Message for "read hardware status"
        public const int WM_MESSAGE_ERROR = WM_USER + 3;             // Message "On Error"

    }



    class cl_Irinos_Base : cl_MscDll
    {
        public const int MSC_BUF_SIZE_STATIC = 1500;    // Size of the Buffer for reading static measurement values
        public const int MSC_BUF_SIZE_BITIO = 64;       // Size of the Buffers for exchanging Bit I/O data
        public const int MSC_BUF_SIZE_HARDSTAT = 256;   // Size of the Buffer for reading the hardware status
        public const int MSC_BUF_SIZE_WRITECMD = 1000;  // Size of the Buffer for exchanging commands with MSC_WriteCommand
        public const int MAX_DYNAMIC_MEASUREMENT = 2;   // Max. number of simultaneous dynamic measurements
        public const int MAX_DYN_CHANNELS = 32;         // Max. number of measurement channels per dynamic measurement

        /* Device-Handle for the connection to the Irinos-System via the MscDll.
         * It is assigned by the MscDll when the connection to the Irinos-System
         * is established (see MSC_OpenDevice). */
        private IntPtr pDevice;

        /* Current connection status. See type definition. */
        public E_CONNECT_STATUS eConnStat = E_CONNECT_STATUS.cscUndefined;

        // Allocate buffers for various functions. The buffers must be fixed, since 
        // the DLL gets pointers to memory addresses. For this purpose, 
        // the Marshal class of the .NET framework is used.
        private IntPtr SndBufferCmd = Marshal.AllocHGlobal(MSC_BUF_SIZE_WRITECMD);
        private IntPtr RcvBufferCmd = Marshal.AllocHGlobal(MSC_BUF_SIZE_WRITECMD);
        private IntPtr SndBufferRS = Marshal.AllocHGlobal(1);
        private IntPtr RcvBufferRS = Marshal.AllocHGlobal(MSC_BUF_SIZE_STATIC);
        private IntPtr SndBufferBIO = Marshal.AllocHGlobal(MSC_BUF_SIZE_BITIO);
        private IntPtr RcvBufferBIO = Marshal.AllocHGlobal(MSC_BUF_SIZE_BITIO * 2);
        private IntPtr SndBufferRHS = Marshal.AllocHGlobal(1);
        private IntPtr RcvBufferRHS = Marshal.AllocHGlobal(MSC_BUF_SIZE_HARDSTAT);
        private IntPtr SndBufferDummy = Marshal.AllocHGlobal(1);
        private IntPtr[,] aValueBuffersDyn = new IntPtr[MAX_DYNAMIC_MEASUREMENT, MAX_DYN_CHANNELS];
        private int[] aiDynBytesPerChannel = new int[2] { 0, 0 };

        /* Multimedia timer, used for lowering the windows system tick time, which
         * speeds up the data transfer rate between the Irinos-System and the MscDll. */
        CMMTimer MMTimer = new CMMTimer();

        
        /******************************************************************************
        * FUNCTION: GetMscDllVersion
        *-----------------------------------------------------------------------------
        * Read the version information from the MscDll.
        * This function calls the DLL-function with try-catch. It is recommended
        * to call this function at the beginning of your application in order to
        * check, if the MscDll can be accessed.
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * iApiMajor: Major version of the MscDll API will be written into this parameter.
        * iApiMinor: Minor version of the MscDll API will be written into this parameter.
        * iDllMajor: Major version of the MscDll will be written into this parameter.
        * iDllMinor: Minor version of the MscDll will be written into this parameter.
        *-----------------------------------------------------------------------------
        * Return: true -> MscDll can be accessed
        *         false -> Failed to access MscDll
        ******************************************************************************/
        public bool GetMscDllVersion(ref int iApiMajor, ref int iApiMinor, ref int iDllMajor, ref int iDllMinor)
        {
            bool bReturnValue = false;
            UInt32 ApiVers = 0;
            UInt32 DllVers = 0;

            try
            {
                F_MSC_GetVersion(ref ApiVers, ref DllVers);
                iApiMajor = (int)(ApiVers >> 16) & 0x0000FFFF;
                iApiMinor = (int)(ApiVers) & 0x0000FFFF;
                iDllMajor = (int)(DllVers >> 16) & 0x0000FFFF;
                iDllMinor = (int)(DllVers) & 0x0000FFFF;
                bReturnValue = true;
            }
            catch
            {
            }

            return bReturnValue;
        }


        /******************************************************************************
        * FUNCTION: ConnectIrinos
        *-----------------------------------------------------------------------------
        * Trying to establish a connection to the first Irinos-System, which can be
        * found. Communication parameters are taken from Msc.cfg by the DLL.
        *
        * Note:
        * It is possible to access more than one Irinos-System from the same PC.
        * In this case more than one Device-Handle must be created.
        * However: In more than 10 years, there was not any application with this
        * requirement.
        *-----------------------------------------------------------------------------
        * PARAMETERS: none
        *-----------------------------------------------------------------------------
        * Return: see type definition
        ******************************************************************************/
        public E_CONNECT_STATUS ConnectIrinos()
        {
            E_CONNECT_STATUS eResult = E_CONNECT_STATUS.cscTryConnect;
            UInt32 ulNDevices = 0;
            UInt32 ulBusType = 0;
            byte[] acStr = new byte[MSC_MAX_UNIQUEID_SIZE];

            /* Get count of available devices.
             * If 1 Irinos-System is connected, 1 should be returned. */
            if (F_MSC_EnumerateDevices(ref ulNDevices) != MSC_STATUS.MSC_STATUS_SUCCESS)
            {
                eResult = E_CONNECT_STATUS.cscNoDevices;
            }

            /* Read device information */
            if (eResult == E_CONNECT_STATUS.cscTryConnect)
            {
                ulBusType = 0;
                if (F_MSC_GetDeviceInfo(ulNDevices - 1, ref ulBusType, acStr) != MSC_STATUS.MSC_STATUS_SUCCESS)
                {
                    eResult = E_CONNECT_STATUS.cscNoDeviceInfo;
                }
            }

            /* Open device / create handle for device */
            if (eResult == E_CONNECT_STATUS.cscTryConnect)
            {
                if (F_MSC_OpenDevice(ulNDevices - 1, ref pDevice) == MSC_STATUS.MSC_STATUS_SUCCESS)
                {
                    eResult = E_CONNECT_STATUS.cscConnected;
                }
                else
                {
                    eResult = E_CONNECT_STATUS.cscNoDeviceOpen;
                }
            }

            /* Initialize device */
            if (eResult == E_CONNECT_STATUS.cscConnected)
            {
                if (F_MSC_InitDevice(pDevice) == MSC_STATUS.MSC_STATUS_SUCCESS)
                {
                    eResult = E_CONNECT_STATUS.cscInitDone;
                }
                else
                {
                    eResult = E_CONNECT_STATUS.cscInitFailed;
                }
            }

            /* Start communication */
            if (eResult == E_CONNECT_STATUS.cscInitDone)
            {
                if (F_MSC_Start(pDevice, 1, 500, 10, 75) == MSC_STATUS.MSC_STATUS_SUCCESS)
                {
                    eResult = E_CONNECT_STATUS.cscCommStarted;
                }
                else
                {
                    eResult = E_CONNECT_STATUS.cscCommFailed;
                }
            }

            /* Start a multimedia timer.
             * This timer is not used by the application. But it lowers the Windows
             * system tick time down to 1ms, which speeds up the communication
             * with the Irinos-System. */
            if (MMTimer.IsRunning == false)
            {
                MMTimer.Resolution = 1;
                MMTimer.Period = 1000;
                MMTimer.Start();
            }

            eConnStat = eResult;
            return eResult;
        }


        /******************************************************************************
        * FUNCTION: StopAndClose
        *-----------------------------------------------------------------------------
        * Close the connection (if opened).
        * This function should be called before closing the application.
        *-----------------------------------------------------------------------------
        * PARAMETERS: none
        *-----------------------------------------------------------------------------
        * Return: true if a connection was closed.
        ******************************************************************************/
        public bool StopAndClose()
        {
            if (pDevice != IntPtr.Zero)
            {
                F_MSC_Stop(pDevice);
                F_MSC_CloseDevice(pDevice);
                eConnStat = E_CONNECT_STATUS.cscConnectionClosed;
                pDevice = IntPtr.Zero;
                return true;
            }
            else
            {
                return false;
            }
        }


        /******************************************************************************
        * FUNCTION: SetupStaticChannel
        *-----------------------------------------------------------------------------
        * Start reading static measurement values / Bit I/O / hardware status
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * cOpode: DLL-Opcode opcRS or opcBIO or opcRHS.
        * SendSize: Size of data to be sent from corresponding send buffer.
        *           --> currently only required for opcBIO!
        * WindowHandle: Window handle which shall receive the notification message,
        *               when the static channel was updated.
        *-----------------------------------------------------------------------------
        * Return: true if a static channel was setup successfully.
        ******************************************************************************/
        public bool SetupStaticChannel(E_MSC_OPCODES Opcode, UInt32 SendSize, IntPtr WindowHandle)
        {
            bool bReturnValue = false;

            if (eConnStat == E_CONNECT_STATUS.cscCommStarted)
            {
                bReturnValue = true;

                switch (Opcode)
                {
                    case E_MSC_OPCODES.opcRS:
                        {
                            // Register static channel
                            if (F_MSC_SetupStaticChannel(pDevice, (byte)Opcode, 1, (IntPtr)SndBufferRS, MSC_BUF_SIZE_STATIC) != MSC_STATUS.MSC_STATUS_SUCCESS)
                            {
                                bReturnValue = false;
                            }
                            // Register Message "On Response"
                            else if (F_MSC_SetNotificationMessage(pDevice, (int)Opcode, WindowHandle, (UInt32)cl_IriConst.WM_MESSAGE_MSC_READSTATIC, 10, 10) != MSC_STATUS.MSC_STATUS_SUCCESS)
                            {
                                bReturnValue = false;
                            }
                            // Register Message for "On Error"
                            else if (F_MSC_SetNotificationMessage(pDevice, -1, WindowHandle, (UInt32)cl_IriConst.WM_MESSAGE_ERROR, 10, 10) != MSC_STATUS.MSC_STATUS_SUCCESS)
                            {
                                bReturnValue = false;
                            }
                        }
                        break;

                    case E_MSC_OPCODES.opcBIO:
                        {
                            // Initialize output buffer
                            byte[] Buf0 = new byte[MSC_BUF_SIZE_BITIO];
                            Array.Clear(Buf0, 0, 0);
                            Marshal.Copy(Buf0, 0, SndBufferBIO, MSC_BUF_SIZE_BITIO);

                            if (SendSize > MSC_BUF_SIZE_BITIO)
                            {
                                SendSize = (UInt32)MSC_BUF_SIZE_BITIO;
                            }
                            // Register static channel
                            if (F_MSC_SetupStaticChannel(pDevice, (byte)Opcode, SendSize, SndBufferBIO, MSC_BUF_SIZE_BITIO * 2) != MSC_STATUS.MSC_STATUS_SUCCESS)
                            {
                                bReturnValue = false;
                            }
                            // Register Message "On Response"
                            else if (F_MSC_SetNotificationMessage(pDevice, (int)Opcode, WindowHandle, (UInt32)cl_IriConst.WM_MESSAGE_MSC_BITIO, 10, 10) != MSC_STATUS.MSC_STATUS_SUCCESS)
                            {
                                bReturnValue = false;
                            }
                        }
                        break;

                    case E_MSC_OPCODES.opcRHS:
                        {
                            byte[] SendData = new byte[1] { 2 };
                            Marshal.Copy(SendData, 0, SndBufferRHS, 1);
                            // Register static channel
                            if (F_MSC_SetupStaticChannel(pDevice, (byte)Opcode, 1, SndBufferRHS, MSC_BUF_SIZE_HARDSTAT) != MSC_STATUS.MSC_STATUS_SUCCESS)
                            {
                                bReturnValue = false;
                            }
                            // Register Message "On Response"
                            else if (F_MSC_SetNotificationMessage(pDevice, (int)Opcode, WindowHandle, (UInt32)cl_IriConst.WM_MESSAGE_MSC_HW_STATUS, 10, 10) != MSC_STATUS.MSC_STATUS_SUCCESS)
                            {
                                bReturnValue = false;
                            }
                        }
                        break;

                    default:
                        {
                            bReturnValue = false;
                        }
                        break;
                }
            }

            return bReturnValue;
        }


        /******************************************************************************
        * FUNCTION: GetStaticMeasurementValues
        *-----------------------------------------------------------------------------
        * Copy the static measurement values into the buffer given in the parameters.
        * This function is typically called after new measurement values have been
        * received, e.g. to update the GUI.
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * aslStaticValues: New static measurement values will be copied into this
        *                  array.
        *-----------------------------------------------------------------------------
        * Return: true if new values were copied
        ******************************************************************************/
        public bool GetStaticMeasurementValues(ref Int32[] aslStaticValues)
        {
            bool bReturnValue = false;
            UInt32 ulDataCount = 0;

            if (eConnStat == E_CONNECT_STATUS.cscCommStarted)
            {
                // Request values from the DLL.
                if (F_MSC_ReadStatic(pDevice, (byte)E_MSC_OPCODES.opcRS, MSC_BUF_SIZE_STATIC, RcvBufferRS, ref ulDataCount) == MSC_STATUS.MSC_STATUS_SUCCESS)
                {
                    if (ulDataCount > (aslStaticValues.Length * 4))
                    {
                        ulDataCount = (UInt32)aslStaticValues.Length * 4;
                    }
                    Marshal.Copy(RcvBufferRS, aslStaticValues, 0, (int)ulDataCount / 4);
                    bReturnValue = true;
                }
            }

            return bReturnValue;
        }


        /******************************************************************************
        * FUNCTION: GetHardwareStatusData
        *-----------------------------------------------------------------------------
        * Copy the hardware status data into the buffer given in the parameters.
        * This function is typically called after new hardware status data has been
        * received.
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * aucHardStat: New hardware status data will be copied into this array.
        *-----------------------------------------------------------------------------
        * Return: true if new values were copied
        ******************************************************************************/
        public bool GetHardwareStatusData(ref byte[] aucHardStat)
        {
            bool bReturnValue = false;
            UInt32 ulDataCount = 0;

            if (eConnStat == E_CONNECT_STATUS.cscCommStarted)
            {
                // Request hardware status data from the DLL.
                if (F_MSC_ReadStatic(pDevice, (byte)E_MSC_OPCODES.opcRHS, MSC_BUF_SIZE_HARDSTAT, RcvBufferRHS, ref ulDataCount) == MSC_STATUS.MSC_STATUS_SUCCESS)
                {
                    if (ulDataCount > aucHardStat.Length)
                    {
                        ulDataCount = (UInt32)aucHardStat.Length;
                    }
                    Marshal.Copy(RcvBufferRHS, aucHardStat, 0, (int)ulDataCount);
                    bReturnValue = true;
                }
            }

            return bReturnValue;
        }


        /******************************************************************************
        * FUNCTION: GetBitInputs
        *-----------------------------------------------------------------------------
        * Copy the bit input data into the buffer given in the parameters.
        * This function is typically called after new Bit I/O data has been
        * received.
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * abInputs: New input bits will be copied into this array
        *           (1 input Bit per array element).
        *-----------------------------------------------------------------------------
        * Return: true if new values were copied
        ******************************************************************************/
        public bool GetBitInputs(ref bool[] abInputs)
        {
            bool bReturnValue = false;
            UInt32 ulDataCount = 0;
            int targetIndex = 0;
            byte bitOr = 0;


            if (eConnStat == E_CONNECT_STATUS.cscCommStarted)
            {
                // Request received Bit I/O data from the DLL.
                if (F_MSC_ReadStatic(pDevice, (byte)E_MSC_OPCODES.opcBIO, MSC_BUF_SIZE_BITIO * 2, RcvBufferBIO, ref ulDataCount) == MSC_STATUS.MSC_STATUS_SUCCESS)
                {
                    if ((ulDataCount * 8 / 2) > abInputs.Length)
                    {
                        ulDataCount = (UInt32)abInputs.Length / 8;
                    }

                    /* Extract bits from received data */
                    byte[] aucTempBuf = new byte[ulDataCount];
                    Marshal.Copy(RcvBufferBIO, aucTempBuf, 0, (int)ulDataCount);
                    for (int i = (int)(ulDataCount / 2); i < ulDataCount; i++)
                    {
                        for (int j = 0; j < 8; j++)
                        {
                            bitOr = (byte)(1 << j);
                            if ((aucTempBuf[i] & bitOr) == bitOr)
                            {
                                abInputs[targetIndex] = true;
                            }
                            else
                            {
                                abInputs[targetIndex] = false;
                            }
                            targetIndex++;
                        }
                    }
                    bReturnValue = true;
                }
            }

            return bReturnValue;
        }


        /******************************************************************************
        * FUNCTION: RefreshBitOutputs
        *-----------------------------------------------------------------------------
        * Copy the bit output data from the buffer given in the parameters to the DLL
        * buffer.
        * This function is typically called if Bit I/O data has changed
        * (or cyclically).
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * abOutputs: New output bits to write to the Irinos-System
        *            (1 output Bit per array element).
        *-----------------------------------------------------------------------------
        * Return: true if new output were copied successfully
        ******************************************************************************/
        public bool RefreshBitOutputs(bool[] abOutputs)
        {
            bool bReturnValue = false;

            if (eConnStat == E_CONNECT_STATUS.cscCommStarted)
            {
                int iLen = abOutputs.Length / 8;
                if ((abOutputs.Length % 8) != 0)
                {
                    iLen++;
                }
                if (iLen > MSC_BUF_SIZE_BITIO)
                {
                    iLen = MSC_BUF_SIZE_BITIO;
                }

                int maxOutputsToCopy = abOutputs.Length;
                if ((iLen * 8) < maxOutputsToCopy)
                {
                    maxOutputsToCopy = iLen * 8;
                }

                byte[] aucBufOut = new byte[iLen];
                Array.Clear(aucBufOut, 0, 0);

                for (int i = 0; i < maxOutputsToCopy; i++)
                {
                    byte bitOr = 0;
                    if (abOutputs[i] == true)
                    {
                        bitOr = (byte)(1 << (i % 8));
                    }

                    aucBufOut[i / 8] |= bitOr;
                }

                Marshal.Copy(aucBufOut, 0, SndBufferBIO, iLen);
                if (F_MSC_RefreshChannel(pDevice, (byte)E_MSC_OPCODES.opcBIO) == MSC_STATUS.MSC_STATUS_SUCCESS)
                {
                    bReturnValue = true;
                }
            }

            return bReturnValue;
        }


        /******************************************************************************
        * FUNCTION: WriteCommand
        *-----------------------------------------------------------------------------
        * Write a command to the Irinos-system (and receive the corresponding
        * response).
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * Opcode: Opcode of command, which shall be sent. E.g. opcRMI to read
        *         module information.
        * SendData: Data, which will be sent.
        * ReceiveData: Receive buffer
        * ReceiveSize: Number of bytes received
        *-----------------------------------------------------------------------------
        * Return: true if Command was executed successfully.
        ******************************************************************************/
        public bool WriteCommand(E_MSC_OPCODES Opcode, byte[] SendData, ref byte[] ReceiveData, ref UInt32 ReceiveSize)
        {
            bool bReturnValue = false;
            int SendSize = 0;

            if (eConnStat == E_CONNECT_STATUS.cscCommStarted)
            {
                // Make sure that only the allowed data length is copied into the send buffer
                SendSize = SendData.Length;
                if (SendSize > MSC_BUF_SIZE_WRITECMD)
                {
                    SendSize = MSC_BUF_SIZE_WRITECMD;
                }

                // Copy send data into a buffer with fixed address
                Marshal.Copy(SendData, 0, SndBufferCmd, SendSize);

                // Send the data to the PC and receive the response data
                if (F_MSC_WriteCommand(pDevice, (byte)Opcode, (UInt32)SendSize, (IntPtr)SndBufferCmd, MSC_BUF_SIZE_WRITECMD, (IntPtr)RcvBufferCmd, ref ReceiveSize, 500) == MSC_STATUS.MSC_STATUS_SUCCESS)
                {
                    bReturnValue = true;
                    if (ReceiveSize > ReceiveData.Length)
                    {
                        ReceiveSize = (UInt32)ReceiveData.Length;
                    }
                    /* Copy the response data from the internal receive buffer (which
                     * has a fixed address) into the Buffer ReceiveData given in the
                     * paremeters. */
                    Marshal.Copy(RcvBufferCmd, ReceiveData, 0, (int)ReceiveSize);
                }
            }

            if (bReturnValue == false)
            {
                ReceiveSize = 0;
            }

            return bReturnValue;
        }


        /******************************************************************************
        * FUNCTION: WriteCommandStr
        *-----------------------------------------------------------------------------
        * Write a string command to the Irinos-system (and receive the corresponding
        * response-string).
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * Opcode: Opcode of command, which shall be sent. E.g. opcRMI to read
        *         module information.
        * str: String to send / the receive string will be copied into this variable
        *-----------------------------------------------------------------------------
        * Return: true if Command was executed successfully.
        ******************************************************************************/
        public bool WriteCommandStr(E_MSC_OPCODES Opcode, ref string str)
        {
            bool bReturnValue = false;
            byte[] Cmd0 = new byte[MSC_BUF_SIZE_WRITECMD];
            byte[] byteData;
            int TxLength = 0;
            UInt32 RxLength = 0;

            if (eConnStat == E_CONNECT_STATUS.cscCommStarted)
            {
                // Make sure send and receive buffers are empty
                Array.Clear(Cmd0, 0, MSC_BUF_SIZE_WRITECMD);
                Marshal.Copy(Cmd0, 0, SndBufferCmd, MSC_BUF_SIZE_WRITECMD);
                Marshal.Copy(Cmd0, 0, RcvBufferCmd, MSC_BUF_SIZE_WRITECMD);

                // Convert string to byte array
                byteData = System.Text.Encoding.ASCII.GetBytes(str);
                TxLength = byteData.Length;
                if (TxLength > MSC_BUF_SIZE_WRITECMD)
                {
                    TxLength = MSC_BUF_SIZE_WRITECMD;
                }
                Marshal.Copy(byteData, 0, SndBufferCmd, TxLength);

                // Send command and receive response
                if (WriteCommand(Opcode, byteData, ref Cmd0, ref RxLength) == true)
                {
                    bReturnValue = true;
                    // Convert received data into a string
                    Marshal.Copy(RcvBufferCmd, Cmd0, 0, (int)RxLength);
                    Array.Resize(ref Cmd0, (int)RxLength);
                    str = System.Text.Encoding.ASCII.GetString(Cmd0);
                }
            }

            if (bReturnValue == false)
            {
                str = "";
            }

            return bReturnValue;
        }


        /******************************************************************************
        * FUNCTION: SetupDynamicChannel
        *-----------------------------------------------------------------------------
        * Allocates memory for dynamic measurement values and assigns this buffer
        * to the MscDll. The MscDll will write received dynamic measurement values
        * automatically into this buffer.
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * eOpcode: opcRDM1 or opcRDM2.
        * iMaxMeasValues: Maximum number of measurement values (per channel) to be
        *				   received.
        * iNChannels: Number of measurement channels (max is 32).
        *-----------------------------------------------------------------------------
        * Return: true if dynamic channel was setup successfully.
        ******************************************************************************/
        public bool SetupDynamicChannel(E_MSC_OPCODES opcode, int iMaxSamples, int iNChannelsUsed)
        {
            int iDyn = 0;

            if (eConnStat == E_CONNECT_STATUS.cscCommStarted)
            {
                switch (opcode)
                {
                    case E_MSC_OPCODES.opcRDM1: iDyn = 0; break;
                    case E_MSC_OPCODES.opcRDM2: iDyn = 1; break;
                    default: return false;
                }

                if (iNChannelsUsed > MAX_DYN_CHANNELS)
                {
                    iNChannelsUsed = MAX_DYN_CHANNELS;
                }

                // Setup extended dynamic channel
                if (F_MSC_SetupExtendedDynamicChannel(pDevice, (byte)opcode, (byte)iNChannelsUsed, 1, SndBufferDummy) != MSC_STATUS.MSC_STATUS_SUCCESS)
                {
                    return false;
                }

                // Delete existing old buffers
                for (int iCh = 0; iCh < iNChannelsUsed; iCh++)
                {
                    if (aValueBuffersDyn[iDyn, iCh] != null)
                    {
                        Marshal.FreeHGlobal(aValueBuffersDyn[iDyn, iCh]);
                    }
                }

                // Assign new buffer with required size and hand it over to the MscDll
                // This must be done for each measurement channel used in the dynamic measurement.
                aiDynBytesPerChannel[iDyn] = iMaxSamples * 4;
                for (int iCh = 0; iCh < iNChannelsUsed; iCh++)
                {
                    aValueBuffersDyn[iDyn, iCh] = Marshal.AllocHGlobal(aiDynBytesPerChannel[iDyn]);
                    UInt32 ulSize = (UInt32)aiDynBytesPerChannel[iDyn];
                    for (UInt32 ulB = 0; ulB < ulSize; ulB++)
                    {
                        Marshal.WriteByte(aValueBuffersDyn[iDyn, iCh], 0xFF);
                    }
                    if (F_MSC_AttachSubChannelBuffer(pDevice, (byte)opcode, (byte)iCh, ulSize, aValueBuffersDyn[iDyn, iCh]) != MSC_STATUS.MSC_STATUS_SUCCESS)
                    {
                        return false;
                    }
                }
            }
            else
            {
                return false;
            }

            return true;
        }


        /******************************************************************************
        * FUNCTION: GetDynamicMeasurementValue
        *-----------------------------------------------------------------------------
        * Returns a dynamic measurement value, which has already been received.
        * (Measurement values are stored by the MscDll in aValueBuffersDyn. This
        * buffer is created dynamically in SetupDynamicChannel.)
        * It is NOT checked, if the value has already been received. This should be
        * done using GetDynamicValuesReceived before.
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * eOpcode: opcRDM1 or opcRDM2.
        * iPosition: Sample number starting at 0
        *             (0 -> First sample, 1 -> Second sample, ...).
        * iChannel: Number of measurement channel, starting at 0
        *           (0 -> First channel, 1 -> Second channel, ...).
        * iValue: Value, which is returned. If no value is available, iValue
        *         will not be changed.
        *-----------------------------------------------------------------------------
        * Return: false if the measurement value is outside the buffer range.
        ******************************************************************************/
        public bool GetDynamicMeasurementValue(E_MSC_OPCODES opcode, int iPosition, int iChannelNumber, ref Int32 iValue)
        {
            int iDyn = 0;

            switch (opcode)
            {
                case E_MSC_OPCODES.opcRDM1: iDyn = 0; break;
                case E_MSC_OPCODES.opcRDM2: iDyn = 1; break;
                default: return false;
            }

            try
            {
                iValue = Marshal.ReadInt32(aValueBuffersDyn[iDyn, iChannelNumber], iPosition*4);
                return true;
            }
            catch
            {
                return false;
            }
        }


        /******************************************************************************
        * FUNCTION: GetDynamicValuesReceived
        *-----------------------------------------------------------------------------
        * Retrieves the number of values (and bytes), which have been retrieved
        * for a dynamic measurement so far. Value is always per measurement channel.
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * eOpcode: opcRDM1 or opcRDM2.
        * iNValues: Number of measurement values already received will be written
        *           into this parameter.
        * iBytes: Number of bytes already received will be written into this parameter.
        *-----------------------------------------------------------------------------
        * Return: true if position was read successfully.
        ******************************************************************************/
        public bool GetDynamicValuesReceived(E_MSC_OPCODES opcode, ref int iNValues, ref int iBytes)
        {
            UInt32 ulNBytes = 0;

            if (eConnStat == E_CONNECT_STATUS.cscCommStarted)
            {
                if (F_MSC_GetPosition(pDevice, (byte)opcode, ref ulNBytes) == MSC_STATUS.MSC_STATUS_SUCCESS)
                {
                    iBytes = (int)ulNBytes;
                    iNValues = iBytes / 4;
                    return true;
                }
            }

            return false;
        }
    }
}
