﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace MscDll_Demo_VCS_2017
{
    // Measurement channel types
    public enum E_CHANNEL_TYPES
    {
        UNKNOWN,        // Unknown channel type
        TESA2000,       // Tesa halfbridge ±2000µm or compatible ones
        IET200,         // Knäbel IET ±200µm
        INCTTL,         // Incremental encoder TTL/RS422
        INC1VPP,        // Incremental encoder 1Vpp
        AIND10          // Analogue input ±10V
    };


    /* Measurement channel properties */
    class cl_IriChannel
    {
        public cl_IriChannel()
        {
            strName = "";
            iLogicNumber = 0;
            iBoxAddress = 0;
            iPhysChannel = 0;
            eType = E_CHANNEL_TYPES.UNKNOWN;
            dDigitToUnit = 1.0;
            dUnitToDigit = 1.0;
            iNDecimalPlaces = 1;
            bUseForChannelList = false;
        }

        // Standard information --> always available
        public string strName;      // Name of the channel, e.g. T5
        public int iLogicNumber;    // Logic channel number (in ascending order)
        public int iBoxAddress;     // Address of the Irinos-Box, which provides the measurement channel
        public int iPhysChannel;    // Physical channel number of the measurement channel at the Irinos-Box

        // Extended information --> only available if information is retrieved using GetAvailableChannelsExtended
        public E_CHANNEL_TYPES eType;//Channel type; see type definition
        public double dDigitToUnit; // Factor for converting internal digits to physical unit
        public double dUnitToDigit; // Factor for converting physical unit to digits
        public int iNDecimalPlaces; // Number of decimal places for displaying the physical unit

        // Variables to be set from the application for generating a channel list
        public bool bUseForChannelList; // true if this channel shall be included in a channel list
    }


    // Statusword for dynamic measurement. See documentation of opcode opcRSW.
    class cl_IriDynMeasStatusword
    {
        public cl_IriDynMeasStatusword()
        {
            bTrigger1_Active = false;
            bTrigger1_WasActive = false;
            bTrigger1_Min1Pulse = false;
            bDynMeas1_Active = false;
            bDynMeas1_WasActive = false;
            bDynMeas1_Min1Value = false;
            bDynMeas1_ReadActive = false;
            bDynMeas1_BufferFull = false;
            bTrigger2_Active = false;
            bTrigger2_WasActive = false;
            bTrigger2_Min1Pulse = false;
            bDynMeas2_Active = false;
            bDynMeas2_WasActive = false;
            bDynMeas2_Min1Value = false;
            bDynMeas2_ReadActive = false;
            bDynMeas2_BufferFull = false;
        }

        public bool bTrigger1_Active;       // Trigger 1 is active
        public bool bTrigger1_WasActive;    // Trigger 1 was active and is now inactive
        public bool bTrigger1_Min1Pulse;    // Trigger 1: Minimum 1 trigger pulse occured

        public bool bDynMeas1_Active;       // Dynamic measurement 1 is active
        public bool bDynMeas1_WasActive;    // Dynamic measurement 1 was active and is now inactive
        public bool bDynMeas1_Min1Value;    // Dynamic measurement 1: at least 1 value has been sampled
        public bool bDynMeas1_ReadActive;   // Dynamic measurement 1: reading measurement is active
        public bool bDynMeas1_BufferFull;   // Dynamic measurement 1: Internal buffer full

        public bool bTrigger2_Active;       // Trigger 2 is active
        public bool bTrigger2_WasActive;    // Trigger 2 was active and is now inactive
        public bool bTrigger2_Min1Pulse;    // Trigger 2: Minimum 1 trigger pulse occured

        public bool bDynMeas2_Active;       // Dynamic measurement 2 is active
        public bool bDynMeas2_WasActive;    // Dynamic measurement 2 was active and is now inactive
        public bool bDynMeas2_Min1Value;    // Dynamic measurement 2: at least 1 value has been sampled
        public bool bDynMeas2_ReadActive;   // Dynamic measurement 2: reading measurement is active
        public bool bDynMeas2_BufferFull;   // Dynamic measurement 2: Internal buffer full
    }



    class cl_Irinos : cl_Irinos_Base 
    {
        //#################################################################################
        //#################################################################################
        // This class provides helpful functions for using the Irinos-System,
        // to make it easier for you.
        // It is not required to use these functions: you can also use the functions of the
        // cl_Irinos_Base class directly. 
        //#################################################################################
        //#################################################################################

        
        /******************************************************************************
        * FUNCTION: WriteAbsoluteDateTime
        *-----------------------------------------------------------------------------
        * Write the current data and time to the Irinos-System. This is recommended
        * in order to have date/time information for diagnostic memory entries.
        * --> Communication must be active.
        *-----------------------------------------------------------------------------
        * PARAMETERS: none
        *-----------------------------------------------------------------------------
        * Return: true if writing the current date/time has been successful.
        ******************************************************************************/
        public bool WriteAbsoluteDateTime()
        {
            bool bReturnValue = false;
            DateTime cDt = DateTime.Now;
            string str = "";

            str = "#1;" + cDt.Year.ToString() + ";" +
                          cDt.Month.ToString() + ";" +
                          cDt.Day.ToString() + ";" +
                          cDt.Hour.ToString() + ";" +
                          cDt.Minute.ToString() + ";" +
                          cDt.Second.ToString() + ";" +
                          cDt.Millisecond.ToString() + "#";
            WriteCommandStr(E_MSC_OPCODES.opcSAbsT, ref str);
            if (str == "#0#")
            {
                bReturnValue = true;
            }

            
            return bReturnValue;
        }


        /******************************************************************************
        * FUNCTION: GetAvailableChannelsStandard
        *-----------------------------------------------------------------------------
        * Request a list of all available channels from the Irinos-System with
        * standard information.
        * This function is faster than GetAvailableChannelsExtended, but provides
        * less information.
        * --> Communication must be active.
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * aclChannels: Channel information will be written into this array.
        *              See type definition for more information.
        * strError: In case requesting the list fails, an error text will be
        *           provided via this string.
        *-----------------------------------------------------------------------------
        * Return: true if reading the channel information was successful.
        ******************************************************************************/
        public bool GetAvailableChannelsStandard(ref cl_IriChannel[] aclChannels, ref string strError)
        {
            int iSegIdxReturned = 0;
            int iMaxSegIdx = 1;
            int iHighestChannelNumber = 0;
            string str = "";
            cl_IriChannel[] aclNewCh = new cl_IriChannel[256];

            for (int iSegIdx = 1; (iSegIdx <= iMaxSegIdx); iSegIdx++)
            {
                str = "#" + iSegIdx.ToString() + "#";
                if (WriteCommandStr(E_MSC_OPCODES.opcRCA, ref str) == false)
                {
                    strError = "Communication failure.";
                    return false;
                }

                // Remove leading and trailing # from string.
                str = str.Replace("#", "");

                // Split string into substrings and check if enough substrings are available.
                string[] strSplit = str.Split(';');
                if (strSplit.Length < 3)
                {
                    strError = "Invalid response string from Irinos-System: not enough parameters";
                    return false;
                }

                // Check for a correct segment index
                iSegIdxReturned = 0;
                Int32.TryParse(strSplit[0], out iSegIdxReturned);
                if (iSegIdxReturned != iSegIdx)
                {
                    strError = "Invalid response string from Irinos-System: invalid segment index.";
                    return false;
                }

                // Get maximum segment index
                iSegIdxReturned = 0;
                Int32.TryParse(strSplit[1], out iSegIdxReturned);
                if ( (iSegIdxReturned < 1) || (iSegIdxReturned > 8) )
                {
                    strError = "Invalid response string from Irinos-System: invalid number of segments.";
                    return false;
                }
                else
                {
                    iMaxSegIdx = iSegIdxReturned;
                }

                // Read channel content
                for (int j = 2; j < strSplit.Length; j++)
                {
                    string[] strChInfo = strSplit[j].Split(',');
                    if (strChInfo.Length != 5)
                    {
                        strError = "Invalid response string from Irinos-System: invalid number of channel parameters.";
                        return false;
                    }

                    cl_IriChannel clCh = new cl_IriChannel();
                    clCh.strName = strChInfo[0];
                    if ((Int32.TryParse(strChInfo[1], out clCh.iLogicNumber) == false) ||
                         (Int32.TryParse(strChInfo[2], out clCh.iBoxAddress) == false) ||
                         (Int32.TryParse(strChInfo[4], out clCh.iPhysChannel) == false) ||
                         (clCh.iLogicNumber > 256) || (clCh.iLogicNumber < 1))
                    {
                        strError = "Invalid response string from Irinos-System: invalid channel parameter.";
                        return false;
                    }
                    aclNewCh[clCh.iLogicNumber - 1] = clCh;
                    if (clCh.iLogicNumber > iHighestChannelNumber)
                    {
                        iHighestChannelNumber = clCh.iLogicNumber;
                    }
                }
            }

            // All channel information read successfully. Return channel information.
            Array.Resize(ref aclNewCh, iHighestChannelNumber);
            aclChannels = aclNewCh;

            strError = "";
            return true;
        }


        /******************************************************************************
         * FUNCTION: GetAvailableChannelsExtended
         *-----------------------------------------------------------------------------
         * Request a list of all available channels from the Irinos-System with
         * extended information.
         * This function provides more information than GetAvailableChannelsExtended
         * However, it takes more time to retrieve all the information, especially
         * if many channels are used.
         * --> Communication must be active.
         *-----------------------------------------------------------------------------
         * PARAMETERS:
         * aclChannels: Channel information will be written into this array.
         *              See type definition for more information.
         * strError: In case requesting the list fails, an error text will be
         *           provided via this string.
         *-----------------------------------------------------------------------------
         * Return: true if reading the channel information was successful.
         ******************************************************************************/
        public bool GetAvailableChannelsExtended(ref cl_IriChannel[] aclChannels, ref string strError)
        {
            // First request the standard information.
            if (GetAvailableChannelsStandard(ref aclChannels, ref strError) == false)
            {
                return false;
            }


            // Now request the additional information.
            foreach (cl_IriChannel clCh in aclChannels)
            {
                string str = "#" + clCh.strName + ";1#";
                if (WriteCommandStr(E_MSC_OPCODES.opcRCI, ref str) == false)
                {
                    strError = "Communication failure.";
                    return false;
                }

                // Remove leading and trailing # from string.
                str = str.Replace("#", "");

                // Split string into substrings, each containing one of the string parameters.
                string[] strSplit = str.Split(';');
                if (strSplit.Length < 7)
                {
                    strError = "Invalid response string from Irinos-System: not enough parameters";
                    return false;
                }

                int iNominator = 0;
                int iDenominator = 0;
                if ( (Int32.TryParse(strSplit[3], out iNominator) == false) ||
                     (Int32.TryParse(strSplit[4], out iDenominator) == false) ||
                     (Int32.TryParse(strSplit[6], out clCh.iNDecimalPlaces) == false) ||
                     (iNominator == 0) || (iDenominator == 0))
                {
                    strError = "Invalid channel information received.";
                    return false;
                }

                switch (strSplit[2])
                {
                    case "0x0100": clCh.eType = E_CHANNEL_TYPES.TESA2000; break;
                    case "0x0110": clCh.eType = E_CHANNEL_TYPES.IET200; break;
                    case "0x0200": clCh.eType = E_CHANNEL_TYPES.INCTTL; break;
                    case "0x0210": clCh.eType = E_CHANNEL_TYPES.INC1VPP; break;
                    case "0x0300": clCh.eType = E_CHANNEL_TYPES.AIND10; break;
                    default: clCh.eType = E_CHANNEL_TYPES.UNKNOWN; break;
                }

                clCh.dDigitToUnit = (double)iNominator / (double)iDenominator;
                clCh.dUnitToDigit = 1 / clCh.dDigitToUnit;
            }

            strError = "";
            return true;
        }


        /******************************************************************************
         * FUNCTION: CreateChannelListString
         *-----------------------------------------------------------------------------
         * Creates a channel list string, which can be written to the Irinos-System.
         * For all channels, which shall be contained in the channel list, the
         * flag abUseChannel must be set to true.
         * 
         * Be careful for dynamic measurement: max. 32 channels are allowed. This
         * is not checked.
         * 
         * Alternatively the string can be generated manually and transferred to
         * the Irinos-System using WriteCommandStr together with the Opcode opcWCL.
         * 
         * --> Communication must be active.
         *-----------------------------------------------------------------------------
         * PARAMETERS:
         * iChListNumber: Number of channel list (1..10).
         * aclChannelsUsed: List of channels. For those to be used, the flag
         *                  aclChannelUsed.bUseForChannelList must be set to true.
         *                  The list should have been generated using 
         *                  GetAvailableChannelsXXX before.
         * strChList: Channel list string generated by this function.
         * strError: In case of an error, an error text will be provided via this string.
         *-----------------------------------------------------------------------------
         * Return: true if writing the channel list was successful.
         ******************************************************************************/
        public bool CreateChannelListString(int iChListNumber, bool[] abUseChannel, ref string strChList, ref string strError)
        {
            strChList = "";

            // Check channel list number
            if ((iChListNumber < 1) || (iChListNumber > 10))
            {
                strError = "Invalid channel list number. Valid numbers are 1..10";
                return false;
            }

            // Create string to be written.
            // Example string for 6 channels: #1;T2;T3;T5;T7;T26;T19#
            strChList = "#" + iChListNumber.ToString();
            int i = 1;
            foreach (bool bUse in abUseChannel)
            {
                if (bUse == true)
                {
                    strChList += ";T" + i.ToString();
                }
                i++;
            }

            strChList += "#";
            strError = "";
            return true;
        }


        /******************************************************************************
         * FUNCTION: CreateTriggerDefinition_Time
         *-----------------------------------------------------------------------------
         * Creates a Trigger definition string for time-triggered dynamic measurement.
         * 
         * Alternatively the string can be generated manually and transferred to
         * the Irinos-System using WriteCommandStr together with the Opcode opcDT.
         * 
         * --> Communication must be active.
         *-----------------------------------------------------------------------------
         * PARAMETERS:
         * iTriggerNumber: 1 or 2
         * strSamplePeriodMs: Sample period in MilliSeconds, e.g. 0.1, 0.2, 1.0, 5.0.
         *                    Minimum value is 0.1
         *                    Must be a multiple of 0.05
         * strStartDelayMs: Start delay in MilliSeconds. Most applications do not
         *                  require a start delay, so 0 is used.
         * strEndTime: Time in Milliseconds until the time-triggered dynamic measurement
         *             is stopped. Typically the dynamic measurement is stopped by
         *             limiting the number of samples in the definition of the
         *             dynamic measurement. In this case, use the character * here.
         *-----------------------------------------------------------------------------
         * Return: Trigger definition string.
         ******************************************************************************/
        public string CreateTriggerDefinition_Time(int iTriggerNumber, string strSamplePeriodMs, string strStartDelayMs, string strEndTime)
        {
            return "#" + iTriggerNumber.ToString() + ";T;*;1;" + strSamplePeriodMs + ";" + strStartDelayMs + ";" + strEndTime + "#";
        }


        /******************************************************************************
         * FUNCTION: CreateTriggerDefinition_Position
         *-----------------------------------------------------------------------------
         * Creates a Trigger definition string for position-triggered
         * dynamic measurement.
         * 
         * Alternatively the string can be generated manually and transferred to
         * the Irinos-System using WriteCommandStr together with the Opcode opcDT.
         * 
         * --> Communication must be active.
         *-----------------------------------------------------------------------------
         * PARAMETERS:
         * iTriggerNumber: 1 or 2
         * strTriggerChannel: Name of the channel, which is used as position trigger, e.g.
         *                    T7
         * fScale: Floating point value to convert the measurement value of the
         *         Trigger-Source into a physical unit, e.g. µm or °. This allows using
         *         physical units for the parameters fDistance, fStart and fEnd.
         *         -> Use a scaling factor of 1 to use the measurement value unit
         *            (-> increments or digits).
         *         -> The scaling factor can also be -1 in order to change the sign.
         * fDistance: Distance between 2 trigger-positions. E.g. 1°
         * fStart: The measurement starts after this value has been exceeded.
         * fEnd: The measurement will end, if this value is exceeded. This also applied,
         *       if the maximum number of dynamic measurement values has not been reached
         *       yet.
         *       Alternatively the parameter bUseEnd can be set to false. The
         *       dynamic measurement will then be active, until the maximum number of
         *       dynamic measurement values has been reached (or until it is stopped
         *       manually).
         * bUseEnd: Set to true, if the End-value fEnd shall be used, otherwise to false.
         *-----------------------------------------------------------------------------
         * Return: Trigger definition string.
         ******************************************************************************/
        public string CreateTriggerDefinition_Position(int iTriggerNumber, string strTriggerChannel, float fScale, float fDistance, float fStart, float fEnd, bool bUseEnd)
        {
            if (bUseEnd == true)
            {
                return "#" + iTriggerNumber.ToString() + ";P;" + strTriggerChannel + ";" +
                    fScale.ToString() + ";" + fDistance.ToString() + ";" + fStart.ToString() + ";" + fEnd.ToString() + "#";
            }
            else
            {
                return "#" + iTriggerNumber.ToString() + ";P;" + strTriggerChannel + ";" +
                    fScale.ToString() + ";" + fDistance.ToString() + ";" + fStart.ToString() + ";" + "*#";
            }
        }


        /******************************************************************************
         * FUNCTION: CreateDynamicMeasurementDefinition
         *-----------------------------------------------------------------------------
         * Creates a definition string, for a dynamic measurement.
         * 
         * Alternatively the string can be generated manually and transferred to
         * the Irinos-System using WriteCommandStr together with the Opcode opcDDMx.
         * 
         * --> Communication must be active.
         *-----------------------------------------------------------------------------
         * PARAMETERS:
         * iTriggerNumber: Number of the trigger (1..2) to be used for the measurement.
         * iChannelListNumber: Number of channel list (1..10) to be used for the measurement.
         * bActive: true to enable the dynamic measurement, false to disable it.
         * iMaxSamples: Max number of samples to take.
         * strDynDef: The definition string will be written into this parameter.
         * strError: In case of an error, an error text will be provided via this string.
         *-----------------------------------------------------------------------------
         * Return: true if writing the channel list was successful.
         ******************************************************************************/
        public bool CreateDynamicMeasurementDefinition(int iTriggerNumber, int iChannelListNumber, bool bActive, int iMaxSamples, ref string strDynDef, ref string strError)
        {
            // Check trigger number
            if ((iTriggerNumber < 1) || (iTriggerNumber > 2))
            {
                strError = "Trigger number invalid: " + iTriggerNumber.ToString();
                return false;
            }

            // Check channel list number
            if ((iChannelListNumber < 1) || (iChannelListNumber > 10))
            {
                strError = "Channel list number invalid: " + iTriggerNumber.ToString();
                return false;
            }

            // Generate definition string
            strDynDef = "#;" + iTriggerNumber.ToString() + ";" + iChannelListNumber.ToString();
            if (bActive == true)
            {
                strDynDef += ";1;";
            }
            else
            {
                strDynDef += ";0;";
            }
            strDynDef += iMaxSamples.ToString() + "#";

            strError = "";
            return true;
        }


        /******************************************************************************
         * FUNCTION: GetDynamicMeasurementStatus
         *-----------------------------------------------------------------------------
         * Reads the statusword for dynamic measurement from the Irinos-System.
         * 
         * Alternatively the statusword cyn be requested manually from the
         * Irinos-System using WriteCommand together with the Opcode opcRSW.
         * 
         * --> Communication must be active.
         *-----------------------------------------------------------------------------
         * PARAMETERS: none
         *-----------------------------------------------------------------------------
         * Return: See type definition
         ******************************************************************************/
        public cl_IriDynMeasStatusword GetDynamicMeasurementStatus()
        {
            cl_IriDynMeasStatusword clSw = new cl_IriDynMeasStatusword();
            byte[] SndBuf = new byte[] { 0 };
            byte[] RcvBuf = new byte[] { 0, 0, 0, 0 };
            UInt32 ulRecSize = 0;
            Int32 iSw = 0;

            // Request the statusword from the Irinos-System
            WriteCommand(E_MSC_OPCODES.opcRSW, SndBuf, ref RcvBuf, ref ulRecSize);

            // Convert the 4 bytes into an Int32-variable
            iSw = BitConverter.ToInt32(RcvBuf, 0);

            // Convert the Int32-bitfield variable into single bits
            clSw.bTrigger1_Active = ((iSw & 0x00000001) == 0x00000001);
            clSw.bTrigger1_WasActive = ((iSw & 0x00000002) == 0x00000002);
            clSw.bTrigger1_Min1Pulse = ((iSw & 0x00000004) == 0x00000004);

            clSw.bDynMeas1_Active = ((iSw & 0x00000010) == 0x00000010);
            clSw.bDynMeas1_WasActive = ((iSw & 0x00000020) == 0x00000020);
            clSw.bDynMeas1_Min1Value = ((iSw & 0x00000040) == 0x00000040); ;
            clSw.bDynMeas1_ReadActive = ((iSw & 0x00000080) == 0x00000080);
            clSw.bDynMeas1_BufferFull = ((iSw & 0x00000100) == 0x00000100);

            clSw.bTrigger2_Active = ((iSw & 0x00000001) == 0x00010000);
            clSw.bTrigger2_WasActive = ((iSw & 0x00020000) == 0x00020000);
            clSw.bTrigger2_Min1Pulse = ((iSw & 0x00040000) == 0x00040000);

            clSw.bDynMeas2_Active = ((iSw & 0x00100000) == 0x00100000);
            clSw.bDynMeas2_WasActive = ((iSw & 0x00200000) == 0x00200000);
            clSw.bDynMeas2_Min1Value = ((iSw & 0x00400000) == 0x00400000);
            clSw.bDynMeas2_ReadActive = ((iSw & 0x00800000) == 0x00800000);
            clSw.bDynMeas2_BufferFull = ((iSw & 0x01000000) == 0x01000000);

            return clSw;
        }


        /******************************************************************************
         * FUNCTION: GetDynamicMeasurementCount
         *-----------------------------------------------------------------------------
         * Reads the number of measurement values (per channel), which have been
         * sampled by the dynamic measurement.
         * Between start and end of the dynamic measurement, this value is 0.
         * After the end of the dyanmic measurement, the count is returned.
         * Example:
         * A dynamic measurement at 10000 Samples/s has been stopped after 5.7 seconds.
         * After stopping the dynamic measurement, ulCount will have the va
         *-----------------------------------------------------------------------------
         * PARAMETERS:
         * opcodeRdm: opcRDM1 for first dynamic measurement, opcRDM2 for second.
         * ulCount: On success, current count will be written into this variable.
         *-----------------------------------------------------------------------------
         * Return: true if count has been read successfully
         ******************************************************************************/
        public bool GetDynamicMeasurementCount(E_MSC_OPCODES opcodeRdm, ref UInt32 ulCount)
        {
            byte[] SndBuf = new byte[] { 0 };
            byte[] RcvBuf = new byte[] { 0, 0, 0, 0, 0, 0, 0, 0 };
            UInt32 ulRecSize = 0;

            if (WriteCommand(E_MSC_OPCODES.opcRDC, SndBuf, ref RcvBuf, ref ulRecSize) == true)
            {
                if (opcodeRdm == E_MSC_OPCODES.opcRDM1)
                {
                    ulCount = BitConverter.ToUInt32(RcvBuf, 0);
                    return true;
                }
                else if (opcodeRdm == E_MSC_OPCODES.opcRDM2)
                {
                    ulCount = BitConverter.ToUInt32(RcvBuf, 4);
                    return true;
                }
            }

            return false;
        }

        /******************************************************************************
        * FUNCTION: DynamicMeasurement_IsTransferFinished
        *-----------------------------------------------------------------------------
        * Provides the information, whether all measurement samples of a dynamic
        * measurement have been transferred to the PC.
        *-----------------------------------------------------------------------------
        * PARAMETERS:
        * opcodeRdm: opcRDM1 for first dynamic measurement, opcRDM2 for second.
        *-----------------------------------------------------------------------------
        * Return: true if all samples have been transferred to the PC.
        * While dynamic measurement is active, false is returned.
        ******************************************************************************/
        public bool DynamicMeasurement_IsTransferFinished(E_MSC_OPCODES opcodeRdm)
        {
            UInt32 ulCount = 0;
            int iNValues = 0;
            int iBytes = 0;

            if ( (GetDynamicValuesReceived(opcodeRdm, ref iNValues, ref iBytes) == true) &&
                 (GetDynamicMeasurementCount(opcodeRdm, ref ulCount) == true) )
            {
                if ( (ulCount != 0) && (ulCount == iNValues) )
                {
                    return true;
                }
            }

            return false;
        }
    }
}
