#include "StdAfx.h"
#include "cl_Irinos.h"
#include <string.h>
#include <stdio.h>


cl_Irinos::cl_Irinos(void)
{
	unsigned long ulI = 0;

	eConnStat = cscUndefined;
	memset(aucRcvBufferStatic, 0, sizeof(aucRcvBufferStatic));
	memset(aucRcvBufferBitIO, 0, sizeof(aucRcvBufferBitIO));
	memset(aucSndBufferBitIO, 0, sizeof(aucSndBufferBitIO));
	memset(aucRcvBufferHardStat, 0, sizeof(aucRcvBufferHardStat));
	memset(aucRcvBufferCmd, 0, sizeof(aucRcvBufferCmd));
	memset(aucSndBufferCmd, 0, sizeof(aucSndBufferCmd));

	for (ulI = 0; ulI < MAX_DYNAMIC_MEASUREMENT; ulI++)
	{
		apslValBufDyn[ulI] = NULL;
		aulSizeValBufDyn[ulI] = 0;
		aulDynBytesPerChannel[ulI] = 0;
	}
}


/******************************************************************************
* FUNCTION: cl_Irinos::ConnectIrinos
*-----------------------------------------------------------------------------
* Trying to establish a connection to the first Irinos-System, which can be
* found. Communication parameters are taken from Msc.cfg by the DLL.
*
* Note:
* It is possible to access more than one Irinos-System from the same PC.
* In this case more than one Device-Handle must be created.
* However: In more than 10 years, there was not any application with this
* requirement.
*-----------------------------------------------------------------------------
* PARAMETERS: none
*-----------------------------------------------------------------------------
* Return: see type definition
******************************************************************************/
E_CONNECT_STATUS cl_Irinos::ConnectIrinos(void)
{
	E_CONNECT_STATUS eResult = cscTryConnect;
	unsigned long ulNDevices = 0;
	unsigned long ulBusType = 0;
	char acStr[MSC_MAX_UNIQUEID_SIZE] = {0};

	/* Get count of available devices.
	 * If 1 Irinos-System is connected, 1 should be returned. */
	if (F_MSC_EnumerateDevices(&ulNDevices) != MSC_STATUS_SUCCESS)
	{
		eResult = cscNoDevices;
	}

	/* Read device information */
	if (eResult == cscTryConnect)
	{
		ulBusType = 0;
		if (F_MSC_GetDeviceInfo(ulNDevices-1, &ulBusType, &acStr[0]) != MSC_STATUS_SUCCESS)
		{
			eResult = cscNoDeviceInfo;
		}
	}

	/* Open device / create handle for device */
	if (eResult == cscTryConnect)
	{
		pDevice = NULL;
		if (F_MSC_OpenDevice(ulNDevices-1, &pDevice) == MSC_STATUS_SUCCESS)
		{
			eResult = cscConnected;
		}
		else
		{
			eResult = cscNoDeviceOpen;
		}
	}

	/* Initialize device */
	if (eResult == cscConnected)
	{
		if (F_MSC_InitDevice(pDevice) == MSC_STATUS_SUCCESS)
		{
			eResult = cscInitDone;
		}
		else
		{
			eResult = cscInitFailed;
		}
	}

	/* Start communication */
	if (eResult == cscInitDone)
	{
		if (F_MSC_Start(pDevice, 1, 500, 10, 75) == MSC_STATUS_SUCCESS)
		{
			eResult = cscCommStarted;
		}
		else
		{
			eResult = cscCommFailed;
		}
	}

	eConnStat = eResult;

	return eResult;
}


/******************************************************************************
* FUNCTION: cl_Irinos::StopAndClose
*-----------------------------------------------------------------------------
* Close the connection (if opened).
* This function should be called before closing the application.
*-----------------------------------------------------------------------------
* PARAMETERS: none
*-----------------------------------------------------------------------------
* Return: true if a connection was closed.
******************************************************************************/
bool cl_Irinos::StopAndClose(void)
{
	if (pDevice != NULL)
	{
		F_MSC_Stop(pDevice);
		F_MSC_CloseDevice(pDevice);
		eConnStat = cscConnectionClosed;
		return true;
	}
	else
	{
		return false;
	}
}



/******************************************************************************
* FUNCTION: cl_Irinos::SetupStaticChannel
*-----------------------------------------------------------------------------
* Start reading static measurement values / Bit I/O / hardware status
*-----------------------------------------------------------------------------
* PARAMETERS:
* ucOpode: DLL-Opcode opcRS or opcBIO or opcRHS.
* ulSendSize: Size of data to be sent from corresponding send buffer.
*             --> currently only required for opcBIO!
* pWindowHandle: Window handle which shall receive the notification message,
*                when the static channel was updated.
*-----------------------------------------------------------------------------
* Return: true if a static channel was setup successfully.
******************************************************************************/
bool cl_Irinos::SetupStaticChannel(E_MSC_OPCODES eOpcode, unsigned long ulSendSize, HWND pWindowHandle)
{
	bool bReturnValue = false;
	static unsigned char ucMeasValSend = 0;
	static unsigned char ucHardStatSend = 2;

	if (this->eConnStat == cscCommStarted)
	{
		bReturnValue = true;

		switch (eOpcode)
		{
			/* ## Read static measurement values */
			case opcRS:
				/* Register static channel */
				if (F_MSC_SetupStaticChannel(pDevice,
					                         (unsigned char)eOpcode,
											 1,
											 &ucMeasValSend,
											 sizeof(aucRcvBufferStatic)) != MSC_STATUS_SUCCESS)
				{
					bReturnValue = false;
				}
				/* Register Message "On Response" */
				else if (F_MSC_SetNotificationMessage(pDevice,
													  (unsigned char)eOpcode,
													  pWindowHandle,
													  WM_MESSAGE_MSC_READSTATIC,
													  10,
													  10) != MSC_STATUS_SUCCESS)
				{
					bReturnValue = false;
				}
				/* Register Message for "On Error" */
				else if (F_MSC_SetNotificationMessage(pDevice,
													  -1,
													  pWindowHandle,
													  WM_MESSAGE_MSC_READSTATIC,
													  20,
													  20) != MSC_STATUS_SUCCESS)
				{
					bReturnValue = false;
				}
			break;

			/* ## Read digital inputs / write digital outputs */
			case opcBIO:
				if (ulSendSize > sizeof(aucSndBufferBitIO))
				{
					ulSendSize = sizeof(aucSndBufferBitIO);
				}

				/* Register static channel */
				if (F_MSC_SetupStaticChannel(pDevice,
											 (unsigned char)eOpcode,
											 ulSendSize,
											 &aucSndBufferBitIO,
											 sizeof(aucRcvBufferBitIO)) != MSC_STATUS_SUCCESS)
				{
					bReturnValue = false;
				}
				/* Register Message "On Response" */
				else if (F_MSC_SetNotificationMessage(pDevice,
													  (unsigned char)eOpcode,
													  pWindowHandle,
													  WM_MESSAGE_MSC_BITIO,
													  10,
													  10) != MSC_STATUS_SUCCESS)
				{
					bReturnValue = false;
				}
			break;

			/* ## Read hardware status */
			case opcRHS:
				/* Register static channel */
				if (F_MSC_SetupStaticChannel(pDevice,
											 (unsigned char)eOpcode,
											 1,
											 &ucHardStatSend,
											 sizeof(aucRcvBufferHardStat)) != MSC_STATUS_SUCCESS)
				{
					bReturnValue = false;
				}
				/* Register Message "On Response" */
				else if (F_MSC_SetNotificationMessage(pDevice,
													  (unsigned char)eOpcode,
													  pWindowHandle,
													  WM_MESSAGE_MSC_HW_STATUS,
													  10,
													  10) != MSC_STATUS_SUCCESS)
				{
					bReturnValue = false;
				}
			break;
			
			/* ## Opcode not supported */
			default:
				bReturnValue = false;
			break;
		}
	}

	return bReturnValue;
}


/******************************************************************************
* FUNCTION: cl_Irinos::GetStaticMeasurementValues
*-----------------------------------------------------------------------------
* Copy the static measurement values into the buffer given in the parameters.
* This function is typically called after new measurement values have been
* received, e.g. to update the GUI.
*-----------------------------------------------------------------------------
* PARAMETERS:
* pslValueBuffer: Static measurement values will be copied into this buffer.
* ulMaxValues: Maximum number of measurement values to copy into the buffer
*              pslValueBuffer. Typically the maximum number of elements
*              in this buffer.
*-----------------------------------------------------------------------------
* Return: nA
******************************************************************************/
void cl_Irinos::GetStaticMeasurementValues(signed long *pslValueBuffer, unsigned long ulMaxValues)
{
	unsigned long ulDataCount = 0;

	if (eConnStat == cscCommStarted)
	{
		/* Request values from the DLL. */
		F_MSC_ReadStatic(pDevice, opcRS, ulMaxValues * sizeof(signed long), (void*)pslValueBuffer, &ulDataCount);
	}
}



/******************************************************************************
* FUNCTION: cl_Irinos::GetBitIO
*-----------------------------------------------------------------------------
* Read Bit input data -> written into aucRcvBufferBitIO.
*-----------------------------------------------------------------------------
* PARAMETERS: nA
*-----------------------------------------------------------------------------
* Return: nA
******************************************************************************/
void cl_Irinos::GetBitIO(void)
{
	unsigned long ulDataCount = 0;

	if (eConnStat == cscCommStarted)
	{
		/* Request values from the DLL. */
		F_MSC_ReadStatic(pDevice, opcBIO, sizeof(aucRcvBufferBitIO), (void*)aucRcvBufferBitIO, &ulDataCount);
	}
}



/******************************************************************************
* FUNCTION: cl_Irinos::RefreshOutputs
*-----------------------------------------------------------------------------
* "Tell" the MscDll that the digital output data (aucSndBufferBitIO) has
* been updated.
* This function must be called after changing the content of aucSndBufferBitIO.
*-----------------------------------------------------------------------------
* PARAMETERS: nA
*-----------------------------------------------------------------------------
* Return: nA
******************************************************************************/
void cl_Irinos::RefreshOutputs(void)
{
	if (eConnStat == cscCommStarted)
	{
		F_MSC_RefreshChannel(pDevice, (unsigned char)opcBIO);
	}
}



/******************************************************************************
* FUNCTION: cl_Irinos::GetBitIO
*-----------------------------------------------------------------------------
* Read hardware status data -> written into aucRcvBufferHardStat.
*-----------------------------------------------------------------------------
* PARAMETERS: nA
*-----------------------------------------------------------------------------
* Return: nA
******************************************************************************/
void cl_Irinos::GetHardStat(void)
{
	unsigned long ulDataCount = 0;

	if (eConnStat == cscCommStarted)
	{
		/* Request values from the DLL. */
		F_MSC_ReadStatic(pDevice, opcRHS, sizeof(aucRcvBufferHardStat), (void*)aucRcvBufferHardStat, &ulDataCount);
	}
}



/******************************************************************************
* FUNCTION: cl_Irinos::WriteCommand
*-----------------------------------------------------------------------------
* Write a command to the Irinos-system (and receive the corresponding
* response).
* Data written to aucSndBufferCmd will be sent.
* Received data will be written to aucRcvBufferCmd.
*-----------------------------------------------------------------------------
* PARAMETERS:
* eOpcode: Opcode of command, which shall be sent. E.g. opcRMI to read
*          module information.
* ulSendSize: Number of bytes of aucSndBufferCmd, which shall be sent.
* pulReceiveSize: Number of received bytes written into aucRcvBufferCmd.
*-----------------------------------------------------------------------------
* Return: true if Command was executed successfully.
******************************************************************************/
bool cl_Irinos::WriteCommand(E_MSC_OPCODES eOpcode, unsigned long ulSendSize, unsigned long *pulReceiveSize)
{
	bool bReturnValue = false;

	if (eConnStat == cscCommStarted)
	{
		/* Make sure ulSendSize is not greater than allowed. */
		if (ulSendSize > sizeof(aucSndBufferCmd))
		{
			ulSendSize = sizeof(aucSndBufferCmd);
		}

		if (F_MSC_WriteCommand(pDevice,
			                   (unsigned char)eOpcode,
							   ulSendSize,
							   &aucSndBufferCmd[0],
							   sizeof(aucRcvBufferCmd),
							   &aucRcvBufferCmd[0],
							   pulReceiveSize,
							   500) == MSC_STATUS_SUCCESS)
		{
			bReturnValue = true;
		}
	}

	if (bReturnValue == false)
	{
		*pulReceiveSize = 0;
	}

	return bReturnValue;
}



/******************************************************************************
* FUNCTION: cl_Irinos::SetupDynamicChannel
*-----------------------------------------------------------------------------
* Allocates memory for dynamic measurement values and assigns this buffer
* to the MscDll. The MscDll will write received dynamic measurement values
* automatically into this buffer.
*-----------------------------------------------------------------------------
* PARAMETERS:
* eOpcode: opcRDM1 or opcRDM2.
* ulMaxMeasValues: Maximum number of measurement values (per channel) to be
*				   received.
* ulNChannels: Number of measurement channels.
*-----------------------------------------------------------------------------
* Return: true if dynamic channel was setup successfully.
******************************************************************************/
bool cl_Irinos::SetupDynamicChannel(E_MSC_OPCODES eOpcode, unsigned long ulMaxMeasValues, unsigned long ulNChannels)
{
	bool bReturnValue = false;
	unsigned long ulSize = ulMaxMeasValues * ulNChannels;
	unsigned long ulBufferSize = ulSize * sizeof(signed long);
	unsigned long ulDyn = 0;
	static unsigned char ucDummy = 0;
	unsigned long ulI = 0;


	if (eConnStat == cscCommStarted)
	{
		if (eOpcode == opcRDM2)
		{
			ulDyn = 1;
		}

		aulDynBytesPerChannel[ulDyn] = ulMaxMeasValues * sizeof(signed long);

		/* Setup extended dynamic channel */
		bReturnValue =  (F_MSC_SetupExtendedDynamicChannel(pDevice,
 														   (unsigned char)eOpcode,
 														   (unsigned char)ulNChannels,
														   1,
														   &ucDummy) == MSC_STATUS_SUCCESS);
		if (bReturnValue == true)
		{
			/* Delete old buffer, if it exists */
			if (apslValBufDyn[ulDyn] != NULL)
			{
				delete[] apslValBufDyn[ulDyn];
			}

			/* Assign new buffer with required size */
			apslValBufDyn[ulDyn] = new signed long[ulBufferSize];
			aulSizeValBufDyn[ulDyn] = ulBufferSize;
		}


		for (ulI = 0; (ulI < ulNChannels) && (bReturnValue == true); ulI++)
		{
			bReturnValue = (F_MSC_AttachSubChannelBuffer(pDevice,
				                                         (unsigned char)eOpcode,
														 (unsigned char)ulI,
														 aulDynBytesPerChannel[ulDyn],
														 (void*)(apslValBufDyn[ulDyn] + (ulI * aulDynBytesPerChannel[ulDyn]))) == MSC_STATUS_SUCCESS);
		}
	}



	return bReturnValue;
}



/******************************************************************************
* FUNCTION: cl_Irinos::GetDynBytesReceived
*-----------------------------------------------------------------------------
* Returns the number of bytes that have been received so far for the dynamic
* measurement given in the opcode.
* Note:
* Each measurement value is a 32 Bit signed value (even if it is 16 or 8 bit
* value inside the Irinos system). To get the number of values received per
* channel, use the following formula:
* nValsPerCchannelReceived = nBytesReceived / 4.
*-----------------------------------------------------------------------------
* PARAMETERS:
* eOpcode: opcRDM1 or opcRDM2.
* ulPosition: Number of bytes received per channel. This value is only
*             changed if the return value is true.
*-----------------------------------------------------------------------------
* Return: true if position was read successfully.
******************************************************************************/
bool cl_Irinos::GetDynBytesReceived(E_MSC_OPCODES eOpcode, unsigned long* pulPosition)
{
	bool bReturnValue = false;
	unsigned long ulValRead = 0;

	if (eConnStat == cscCommStarted)
	{
		bReturnValue = (F_MSC_GetPosition(pDevice, (unsigned char)eOpcode, &ulValRead) == MSC_STATUS_SUCCESS);
		if (bReturnValue == true)
		{
			*pulPosition = ulValRead;
		}
	}

	return bReturnValue;
}



/******************************************************************************
* FUNCTION: cl_Irinos::GetDynValue
*-----------------------------------------------------------------------------
* Returns a dynamic measurement value, which has already been received.
* (Measurement values are stored by the MscDll in apslValBufDyn. This
* buffer is created dynamically in cl_Irinos::SetupDynamicChannel.)
* It is NOT checked, if the value has already been received. This should be
* done using cl_Irinos::GetDynBytesReceived before.
*-----------------------------------------------------------------------------
* PARAMETERS:
* eOpcode: opcRDM1 or opcRDM2.
* ulPosition: Number of the measurement value, beginning at 0.
*             (0 -> First value, 1 -> Second value, ...).
* ulChannel: Number of measurement channel.
* pslValue: Value, which is returned. If no value is available, pslValue
*           will not be changed.
*-----------------------------------------------------------------------------
* Return: false if the measurement value is outside the buffer range.
******************************************************************************/
bool cl_Irinos::GetDynValue(E_MSC_OPCODES eOpcode, unsigned long ulPosition, unsigned long ulChannel, signed long *pslValue)
{
	bool bReturnValue = false;
	unsigned long ulBytePos = 0;
	unsigned long ulDyn = 0;
	signed long *psl = NULL;

	if (eOpcode == opcRDM2)
	{
		ulDyn = 1;
	}

	ulBytePos = ulPosition;
	ulBytePos += ulChannel * aulDynBytesPerChannel[ulDyn];
	
	if ( (apslValBufDyn[ulDyn] != 0) &&
		 ((aulSizeValBufDyn[ulDyn] * sizeof(signed long)) >= (ulBytePos + sizeof(signed long)-1)) )
	{
		psl = (signed long*)(apslValBufDyn[ulDyn] + ulBytePos);
		*pslValue = *psl;
		bReturnValue = true;
	}

	return bReturnValue;
}


