/*
 * Intel QV Linux kernel driver
 * Copyright (c) 1999 - 2013, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/*
 *
 * Module Name:
 *   linuxnaldriver.c
 *
 * Abstract:
 *   This contains the functions necessary for the Linux Driver
 *   component of NAL to function as a driver. This includes the
 *   driver entry point and all Linux specific ring 0 functions.
 *
 *
 * VSS Revision Control Information:
 * ---------------------------------
 *   $Workfile: linuxnaldriver.c $
 *   $Date: 2013/01/09 15:51:36 $
 *   $Archive: /QV2.0/nal/src/linux/driver/linuxnaldriver.c $
 *   $Revision: 1.33 $
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/pci_ids.h>
#include <linux/errno.h>
#include <linux/fs.h>           /* Memory mapping functions */
#include <linux/mm.h>           /* Memory mapping functions */
#include <linux/version.h>
#include <asm/uaccess.h>
#include "naltypes.h"
#include "nalioctl.h"
#include "linuxnaldriver.h"

int
init_module(void);
void
cleanup_module(void);

//module_init(nal_driver_init);
//module_exit(nal_driver_cleanup);

extern spinlock_t        Global_AtomicTestSetSpinLock;
static char              Global_NalDeviceName[] = NAL_LINUX_DRIVER_NAME;
static int               Global_NalMajor        = 0;
NAL_ADAPTER_IN_USE_TABLE Global_AdapterInUse[NAL_DRIVER_MAX_ADAPTERS];
UINT32                   Global_DriverReferenceCount = 0;

char NalLinuxDriverVersion[] = LINUX_DRIVER_VERSION;

/* This is the file operations structure where all the entry point functions are
 * registered with Linux. */
static struct file_operations Global_NalFops =
{
    owner:          THIS_MODULE,
    /* Starting from kernel version 2.6.36, the 'ioctl' is depreciated.
     * Instead of this we must use 'unlocked_ioctl'.
     * See: http://lwn.net/Articles/400074 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,36)
    ioctl:          NalDeviceControl,
#endif
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,10)
    /* Special include for 2.6.11 and later, where we can call an ioctl
     * without calling a lock on the entire kernel. */
    unlocked_ioctl: NalDeviceControlUnlockedIoctl,
#endif
    open:           NalOpen,
    release:        NalRelease
};

/* These macro calls help setup the modinfo fields in the linux ELF file format, which are
*  then used by modinfo command line utility on linux to provide module information to the user */
MODULE_AUTHOR(DRIVER_COMPANYNAME);
MODULE_DESCRIPTION(DRIVER_DESCRIPTION);
MODULE_LICENSE("GPL");
MODULE_VERSION(LINUX_DRIVER_VERSION);

/***************************************************************************
**
** Name:            nal_driver_init()
**
** Description:     Called by Linux when the driver is loaded (insmod).
**
** Author:          MVM
**
** Born on Date:    02/19/2003
**
** Arguments:       None
**
** Returns:         int
**
****************************************************************************/
int
init_module(void)
{
    int     Result  = 0;
    UINT32  i       = 0;

    printk(KERN_DEBUG "Intel Pro Diagnostic Driver loading (v. %s)\n", LINUX_DRIVER_VERSION);
    Result = register_chrdev(0, Global_NalDeviceName, &Global_NalFops);
    if(Result < 0)
    {
        Result = -ENODEV;
    }
    else
    {
        Global_NalMajor = Result;
        Result = 0;
    }

    /* Clear the in use table */
    for(i=0; i<NAL_DRIVER_MAX_ADAPTERS; i++)
    {
        memset(&Global_AdapterInUse[i], 0, sizeof(NAL_ADAPTER_IN_USE_TABLE));
    }

    /* Init the atomic test/set spinlock */
    spin_lock_init(&Global_AtomicTestSetSpinLock);

    return Result;
}

/***************************************************************************
**
** Name:            cleanup_module()
**
** Description:     Called by Linux when the driver is unloaded (rmmod).
**
** Author:          MVM
**
** Born on Date:    02/19/2003
**
** Arguments:       None
**
** Returns:         int
**
****************************************************************************/
void
cleanup_module(void)
{
    printk(KERN_DEBUG "Intel Pro Diagnostic Driver exiting\n");
    unregister_chrdev(Global_NalMajor, Global_NalDeviceName);
}

/***************************************************************************
**
** Name:            NalDeviceControlUnlockedIoctl()
**
** Description:     Called by Linux Kernel when an IOCTL is sent to the driver
**                  without locking the kernel.
**
** Author:          OP
**
** Born on Date:    08/20/2007
**
** Arguments:       Inode  = Inode file descriptor.
**                  File   = File Descriptor file pointer.
**                  Cmd    = Ioctl Number
**                  Arg    = Ioctl Argument
**
** Returns:         int
**
****************************************************************************/
long
NalDeviceControlUnlockedIoctl(
    struct file*    File,
    unsigned int    Cmd,
    unsigned long   Arg
    )

{
    return NalDeviceControl(NULL, File, Cmd, Arg);
}

/***************************************************************************
**
** Name:            NalDeviceControl()
**
** Description:     Called by Linux Kernel when an IOCTL is sent to the driver.
**
** Author:          MVM
**
** Born on Date:    02/26/2003
**
** Arguments:       Inode  = Inode file descriptor.
**                  File   = File Descriptor file pointer.
**                  Cmd    = Ioctl Number
**                  Arg    = Ioctl Argument
**
** Returns:         int
**
****************************************************************************/
int
NalDeviceControl(
    struct inode*   Inode,
    struct file*    File,
    unsigned int    Cmd,
    unsigned long   Arg
    )
{
    NAL_IOCTL_INPUT_DATA  InputData;
    NAL_IOCTL_INPUT_DATA* NalIoctlInputData = &InputData;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    mm_segment_t          old_fs;
#endif
    NAL_STATUS            Status            = NAL_SUCCESS;
    UINT32                IoctlNumber       = 0;
    int                   Result            = 0;


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    /* The following instructions are used to change the bounds used for checking
     * the validity of pointers in copy_from_user. This expands the useable
     * address space to all user and kernel memory. When we are done we revert back
     * to the old bounds. */
    old_fs = get_fs();
    set_fs(get_ds());
    Result = copy_from_user(&InputData, (void*)Arg, sizeof(NAL_IOCTL_INPUT_DATA));
    set_fs(old_fs);
#else
    Result = copy_from_user(&InputData, (void*)Arg, sizeof(NAL_IOCTL_INPUT_DATA));
#endif
    if(Result == 0)
    {
        /* Decode the IOCTL number. This identifies the specific IOCTL. */
        IoctlNumber      = Cmd;

        /* Determine which IOCTL was passed in then call the sub ioctl function for that IOCTL type. */
        switch (IoctlNumber)
        {
        case IOCTL_NAL_OSI:
            Status = NalResolveOsiIoctl(NalIoctlInputData);
            break;

        case IOCTL_NAL_HW_BUS:
            Status = NalResolveHwBusIoctl(NalIoctlInputData);
            break;

        case IOCTL_NAL_NDI:
            Status = NalResolveNdiIoctl(NalIoctlInputData);
            break;

        case IOCTL_NAL_OS_SPECIFIC:
            Status = NalResolveOsSpecificIoctl(NalIoctlInputData);
            break;

        default:
            /* Unrecognized IOCTL handled here */
            Status = NAL_INVALID_PARAMETER;
            Result = EINVAL;
            break;
        }
    }
    if(Result  == 0)
    {
        /* The following instructions are used to change the bounds used for checking
         * the validity of pointers in copy_to_user. This expands the useable
         * address space to all user and kernel memory. When we are done we revert back
         * to the old bounds. */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
        old_fs = get_fs();
        set_fs(get_ds());
        Result = copy_to_user((void*)Arg, NalIoctlInputData, sizeof(NAL_IOCTL_INPUT_DATA));
        set_fs(old_fs);
#else

        Result = copy_to_user((void*)Arg, NalIoctlInputData, sizeof(NAL_IOCTL_INPUT_DATA));
#endif
    }
    return Result;
}

/***************************************************************************
**
** Name:            NalOpen()
**
** Description:     Called by Linux Kernel when an Open request is sent to the driver.
**
** Author:          MVM
**
** Born on Date:    02/26/2003
**
** Arguments:       None
**
** Returns:         int
**
****************************************************************************/
int
NalOpen(
    struct inode* Inode,
    struct file*  File
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            NalRelease()
**
** Description:     Called by Linux Kernel when a Release request is sent to the driver.
**
** Author:          MVM
**
** Born on Date:    02/26/2003
**
** Arguments:       None
**
** Returns:         int
**
****************************************************************************/
int
NalRelease(
    struct inode*   Inode,
    struct file*    File
    )
{
    return 0;
}

/***************************************************************************
**
** Name:            _NalMarkAdapterInUse()
**
** Description:     This attempts to mark the adapter in use if the init flags are not
**                  shared. It returns if the driver was already marked in use and if
**                  the driver can be used.
**
** Author:          MVM
**
** Born on Date:    01/31/2003
**
** Arguments:       NalDevice   = NAL_DEVICE_LOCATION with the PCI or BUS device info.
**                  Lock        = TRUE to lock the device, FALSE to unlock it.
**
** Returns:         TRUE  = Device is in use
**                  FALSE = Device is not yet in use.
**
****************************************************************************/
BOOLEAN
_NalMarkAdapterInUse(
    IN  NAL_DEVICE_LOCATION   NalDevice,
    IN  BOOLEAN               Lock
    )
{
    UINTN   i         = 0;
    BOOLEAN CanBeUsed = FALSE;

    /* First scan all adapters in memory to see if any match the NalDevice requested */
    for(i=0; i<NAL_DRIVER_MAX_ADAPTERS; i++)
    {
        /* Check to see if this adapter slot matches our NalDevice */
        if(Global_AdapterInUse[i].DeviceLocation.Reserved == NalDevice.Reserved)
        {
            /* See if the matching device is already marked "In Use" */
            if(Global_AdapterInUse[i].InUse == TRUE)
            {
                /* The device is in use, determine if we're being asked to "unlock" it */
                if(Lock == FALSE)
                {
                    Global_AdapterInUse[i].InUse = FALSE;
                    Global_AdapterInUse[i].DeviceLocation.Reserved = 0;
                    CanBeUsed = TRUE;
                    break;
                }
                else
                {
                    CanBeUsed = FALSE;
                    break;
                }
            }

            /* Slot found, we're asking to mark in use, and the dev is not in use */
            else if(Lock == TRUE)
            {
                Global_AdapterInUse[i].InUse = TRUE;
                CanBeUsed = TRUE;
                break;
            }
        }
    }

    /* Entry not found and we're looking to add one. */
    if(i == NAL_DRIVER_MAX_ADAPTERS && Lock == TRUE)
    {
        /* Look for a free slot */
        for(i=0; i<NAL_DRIVER_MAX_ADAPTERS; i++)
        {
            if(Global_AdapterInUse[i].DeviceLocation.Reserved == 0)
            {
                break;
            }
        }

        /* This only runs if a free slot was located */
        if(i<NAL_DRIVER_MAX_ADAPTERS)
        {
            Global_AdapterInUse[i].DeviceLocation.Reserved = NalDevice.Reserved;
            Global_AdapterInUse[i].InUse = Lock;
            CanBeUsed = TRUE;
        }
    }

    /* This happens if the adapter is requested to be unlocked but doesnt exist */
    else if(i == NAL_DRIVER_MAX_ADAPTERS && Lock == FALSE)
    {
        CanBeUsed = TRUE;
    }

    return CanBeUsed;
}

/***************************************************************************
**
** Name:            _NalIsAdapterInUse()
**
** Description:     This searches to see if the adapter is already locked as in use.
**
** Author:          MVM
**
** Born on Date:    03/22/2003
**
** Arguments:       NalDevice   = NAL_DEVICE_LOCATION with the PCI or BUS device info.
**
** Returns:         TRUE  = device can be used
**                  FALSE = device cannot be used.
**
****************************************************************************/
BOOLEAN
_NalIsAdapterInUse(
    IN  NAL_DEVICE_LOCATION   NalDevice
    )
{
    UINTN   i         = 0;
    BOOLEAN IsInUse   = FALSE;

    /* First scan all adapters in memory to see if any match the NalDevice requested */
    for(i=0; i<NAL_DRIVER_MAX_ADAPTERS; i++)
    {
        /* Check to see if this adapter slot matches our NalDevice */
        if(Global_AdapterInUse[i].DeviceLocation.Reserved == NalDevice.Reserved)
        {
            /* See if the matching device is already marked "In Use" */
            if(Global_AdapterInUse[i].InUse == TRUE)
            {
                IsInUse = TRUE;
                break;
            }
        }
    }

    return IsInUse;
}


/***************************************************************************
**
** Name:            _NalDriverGetReferenceCount()
**
** Description:     This returns the reference count of the number of user mode components
**                  have referenced the driver.
**
** Author:          SPD
**
** Born on Date:    08/19/2003
**
** Arguments:       None
**
** Returns:         Number of references
**
****************************************************************************/
UINT32
_NalDriverGetReferenceCount(
	VOID
	)
{
    return Global_DriverReferenceCount;
}

/***************************************************************************
**
** Name:            _NalDriverIncrementReferenceCount()
**
** Description:     This increments the reference count of the number of user mode components
**                  have referenced the driver.
**
** Author:          SPD
**
** Born on Date:    08/19/2003
**
** Arguments:       nothing
**
** Returns:         nothing
**
****************************************************************************/
VOID
_NalDriverIncrementReferenceCount(
    VOID
    )
{
    /* Note, do not user the atomic increment here because it's intended for ring3 and will perform
     * a copy_from_user and copy_to_user on the address */
    spin_lock(&Global_AtomicTestSetSpinLock);
    Global_DriverReferenceCount++;
    spin_unlock(&Global_AtomicTestSetSpinLock);
}

/***************************************************************************
**
** Name:            _NalDriverDecrementReferenceCount()
**
** Description:     This decrements the reference count of the number of user mode components
**                  have referenced the driver.
**
** Author:          SPD
**
** Born on Date:    08/19/2003
**
** Arguments:       nothing
**
** Returns:         nothing
**
****************************************************************************/
VOID
_NalDriverDecrementReferenceCount(
    VOID
    )
{
    /* Note, do not user the atomic increment here because it's intended for ring3 and will perform
     * a copy_from_user and copy_to_user on the address */
    spin_lock(&Global_AtomicTestSetSpinLock);
    if(Global_DriverReferenceCount > 0)
    {
        Global_DriverReferenceCount--;
    }
    spin_unlock(&Global_AtomicTestSetSpinLock);
}

/***************************************************************************
**
** Name:            _NalDriverGetVersion()
**
** Description:     This returns the version number of the driver.
**
** Author:          AS
**
** Born on Date:    03/09/2009
**
** Arguments:       None
**
** Returns:         version
**
****************************************************************************/
VOID
_NalDriverGetVersion(
	OUT CHAR* Version
	)
{
    strcpy(Version, NalLinuxDriverVersion);
}

